#pragma once

#include "raw_segment.h"
#include "datetime_helpers.h"

#include <library/cpp/timezone_conversion/convert.h>
#include <util/datetime/base.h>
#include <util/generic/maybe.h>

namespace NRasp {
    class TSegment {
    public:
        TSegment(const TRawSegment& segment, const NDatetime::TSimpleTM& threadStartTM) noexcept
            : RawSegment_(segment)
        {
            const auto& departure = segment.Departure();
            const auto& arrival = segment.Arrival();

            {
                auto naiveDepartureDateTime = AddMinutes(threadStartTM, departure.DepartureOffset().GetRef());

                DepartureDt_ = NDatetime::ToAbsoluteTime(naiveDepartureDateTime, departure.Timezone());
                LocalDepartureTime_ = NDatetime::ToCivilTime(DepartureDt_, segment.DepartureTimezone());
            }
            {
                auto naiveArrivalDateTime = AddMinutes(threadStartTM, arrival.ArrivalOffset().GetRef());

                ArrivalDt_ = NDatetime::ToAbsoluteTime(naiveArrivalDateTime, arrival.Timezone());
                LocalArrivalTime_ = NDatetime::ToCivilTime(ArrivalDt_, segment.ArrivalTimezone());
            }
        }

        TSegment(const TSegment& other)
            : RawSegment_(other.RawSegment_)
            , LocalDepartureTime_(other.LocalDepartureTime_)
            , LocalArrivalTime_(other.LocalArrivalTime_)
            , DepartureDt_(other.DepartureDt_)
            , ArrivalDt_(other.ArrivalDt_)
        {
        }

        inline const TString& ThreadUid() const {
            return RawSegment_.Thread().Uid();
        }

        inline const TThreadStation& Arrival() const noexcept {
            return RawSegment_.Arrival().Item();
        }

        inline const TThreadStationWrapper& ArrivalWrapper() const noexcept {
            return RawSegment_.Arrival();
        }

        inline const TThreadStation& Departure() const noexcept {
            return RawSegment_.Departure().Item();
        }

        inline const TThreadStationWrapper& DepartureWrapper() const noexcept {
            return RawSegment_.Departure();
        }

        inline const TRThread& Thread() const noexcept {
            return RawSegment_.Thread().Item();
        }

        inline const TRThreadWrapper& ThreadWrapper() const noexcept {
            return RawSegment_.Thread();
        }

        inline const TInstant& DepartureDt() const noexcept {
            return DepartureDt_;
        }

        inline const TInstant& ArrivalDt() const noexcept {
            return ArrivalDt_;
        }

        inline ui32 DepartureTimeInDay() const noexcept {
            return LocalDepartureTime().Hour * 60 + LocalDepartureTime().Min;
        }

        inline const NDatetime::TSimpleTM& LocalDepartureTime() const noexcept {
            return LocalDepartureTime_;
        }

        inline const NDatetime::TSimpleTM& LocalArrivalTime() const noexcept {
            return LocalArrivalTime_;
        }

        static const NDatetime::TSimpleTM& GetDepartureTM(const TSegment& segment) {
            return segment.LocalDepartureTime();
        }

        static const NDatetime::TSimpleTM& GetArrivalTM(const TSegment& segment) {
            return segment.LocalArrivalTime();
        }

    private:
        TRawSegment RawSegment_;

        NDatetime::TSimpleTM LocalDepartureTime_;
        NDatetime::TSimpleTM LocalArrivalTime_;

        TInstant DepartureDt_;
        TInstant ArrivalDt_;
    };

    using TSegmentEventTMGetter = std::function<const NDatetime::TSimpleTM&(const TSegment& segment)>;
}
