#pragma once

#include "raw_segment.h"
#include "travel/rasp/route-search-api/point_key.h"
#include "rasp_search_index.h"
#include "segment.h"
#include "travel/rasp/route-search-api/query_reader.h"

#include <util/draft/datetime.h>
#include <util/datetime/parser.h>

namespace NRasp {
    struct TStationSegments {
        TVector<THolder<TSegment>> Departure;
        TVector<THolder<TSegment>> Arrival;
    };

    class TCommonSegmentFinder {
        TVector<THolder<TRawSegment>> FindSegmentsForThread(
            const TVector<const TThreadStationWrapper*>& departures,
            const TVector<const TThreadStationWrapper*>& arrivals,
            const TRThreadWrapper& thread,
            const TLimitConditions& majorityLimit) const;

    public:
        enum class EEventType {
            Departure,
            Arrival
        };

        class TUnknownEventTypeException: public yexception {
        };

        // Ищет сегменты для общих ниток для общих ниток пункта отправления и прибытия
        TVector<THolder<TRawSegment>> Find(const TRaspSearchIndex& searchIndex,
                                           const TPointKey& fromPointKey,
                                           const TPointKey& toPointKey,
                                           const TLimitConditions& limitConditions,
                                           const TString& threadNumber = "") const;

        TVector<THolder<TRawSegment>> FindStationSegments(
            const TRaspSearchIndex& searchIndex,
            const object_id_t stationId,
            const EEventType eventType) const;
    };

    class TDateSegmentFilter {
    public:
        // Для каждого сегмента из segments в ответ попадут лишь те, которые проходят через станцию отправления
        // в период с minDate по maxDate включительно
        TVector<THolder<TSegment>> Filter(
            const TVector<THolder<TRawSegment>>& segments,
            const NDatetime::TSimpleTM minDate,
            const NDatetime::TSimpleTM maxDate,
            const TCommonSegmentFinder::EEventType = TCommonSegmentFinder::EEventType::Departure,
            bool findAll = true,
            int maxThreadSegments = 3) const;
    };

    class TEmptySearchDateFilter {
    public:
        TVector<THolder<TSegment>> Filter(
            TVector<THolder<TSegment>>& segments,
            size_t minCount) const;
    };

    class TMaxDepartureDateFilter {
    public:
        TVector<THolder<TSegment>> Filter(
            TVector<THolder<TSegment>>& segments,
            NDatetime::TSimpleTM minDate) const;
    };

    class TThroughTrainsFilter {
    public:
        // Фильтрует проходящие поезда
        TVector<THolder<TRawSegment>> Filter(
            const TRaspSearchIndex& searchIndex,
            TVector<THolder<TRawSegment>>&& threadSegments) const;
    };

    NDatetime::TSimpleTM CalculateThreadStart(const NDatetime::TTimeZone& timezone,
                                              const NDatetime::TSimpleTM& minEventTM,
                                              ui32 threadStartTime, i64 eventOffset);

    class TSegmentSearcher {
    private:
        TCommonSegmentFinder Finder;
        TThroughTrainsFilter ThroughFilter;
        TDateSegmentFilter DateFilter;
        TEmptySearchDateFilter EmptyDateFilter;
        TMaxDepartureDateFilter MaxDepartureFilter;

    public:
        TSegmentSearcher() = default;

        TVector<THolder<TSegment>> FindDirectionSegments(const TRaspSearchIndex& searchIndex,
                                                         const TDirectionQuery& query) const;

        TVector<THolder<TSegment>> FindEventSegments(const TRaspSearchIndex& searchIndex,
                                                     const TStationQuery& query,
                                                     const TCommonSegmentFinder::EEventType eventType) const;

        TStationSegments FindStationSegments(
            const TRaspSearchIndex& searchIndex,
            const TStationQuery& query) const;
    };

    class TSegmentSearcherWithContext {
    private:
        TSegmentSearcher SegmentSearcher;
        const TRaspSearchIndex& SearchIndex;

    public:
        TSegmentSearcherWithContext(const TSegmentSearcher& searcher, const TRaspSearchIndex& searchIndex)
            : SegmentSearcher(searcher)
            , SearchIndex(searchIndex)
        {
        }

        TVector<THolder<TSegment>> FindDirectionSegments(const TDirectionQuery& query) const {
            return SegmentSearcher.FindDirectionSegments(SearchIndex, query.ToInnerIds(SearchIndex));
        }

        TStationSegments FindStationSegments(const TStationQuery& query) const {
            return SegmentSearcher.FindStationSegments(SearchIndex, query.ToInnerIds(SearchIndex));
        }
    };
}
