#pragma once

#include <travel/rasp/route-search-api/proto/tariff.pb.h>
#include <travel/rasp/route-search-api/proto/currency.pb.h>
#include <library/cpp/langs/langs.h>
#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <travel/rasp/route-search-api/helpers.h>

namespace NRasp {
    class TCurrency {
    private:
        TCurrencyProto::ECurrency Currency_;
        static THashMap<TCurrencyProto::ECurrency, object_id_t> Currencies;

    public:
        TCurrency() = default;

        explicit TCurrency(TCurrencyProto::ECurrency currency)
            : Currency_(currency)
        {
        }

        explicit TCurrency(ELanguage language) {
            switch (language) {
                case LANG_RUS:
                    Currency_ = TCurrencyProto::RUR;
                    break;
                case LANG_UKR:
                    Currency_ = TCurrencyProto::UAH;
                    break;
                default:
                    Clog << __LOCATION__.File << ":" << __LOCATION__.Line
                         << " Trying to get currency for language " << FullNameByLanguage(language) << Endl;
                    Currency_ = TCurrencyProto::RUR;
            }
        }

        bool operator==(const TCurrency& other) const {
            return Currency_ == other.Currency_;
        }

        explicit operator size_t() const {
            return Currencies[Currency_];
        }

        static void AddCurrency(TCurrencyProto::ECurrency currency, object_id_t id);
    };

    static THashMap<ELanguage, THashMap<TCurrency, double>> Rates;

    const THashMap<TCurrency, double>& GetRates(ELanguage nationalVerison);

    class TPrice {
    private:
        TCurrency Currency_;
        double Value_;
        double BaseValue_;

    public:
        TPrice() = default;

        TPrice(double value, TCurrency currency)
            : Currency_(currency)
            , Value_(value)
            , BaseValue_(value)
        {
        }

        TPrice(double value, TCurrencyProto::ECurrency currency)
            : Currency_(currency)
            , Value_(value)
            , BaseValue_(value)
        {
        }

        double Value() const {
            return Value_;
        }

        const TCurrency& Currency() const {
            return Currency_;
        }

        double BaseValue() const {
            return BaseValue_;
        }

        void Rebase(const THashMap<TCurrency, double>& rates, const TCurrency& baseCurrency) {
            if (!rates.contains(baseCurrency))
                return;

            BaseValue_ = Value_ * rates.at(baseCurrency);
        }
    };

    void LoadRates(const TString& source);
}
