#include "title_generator.h"
#include "wrappers.h"
#include "rasp_database.h"
#include "config.h"

TString
NRasp::TDefaultTitleGenerator::PartToText(const NRasp::TSettlementWrapper& pointWrapper, ELanguage lang) const {
    return pointWrapper.Title(lang);
}

TString NRasp::TDefaultTitleGenerator::PartToText(const NRasp::TStationWrapper& pointWrapper, ELanguage lang) const {
    return pointWrapper.Title(lang);
}

TString NRasp::TDefaultTitleGenerator::GenerateTitle(
    const NRasp::TRThreadWrapper& threadWrapper,
    ELanguage lang,
    const TWrappedRaspDatabase& raspDatabase) const {
    TString routeTitle = "";
    bool isFirst = true;

    auto& titleCommon = threadWrapper.Item().title_common();
    const auto& translations = TLanguageConfig::GetTranslations();

    for (auto& part : titleCommon.title_parts()) {
        ui32 settlementId = part.settlement();
        ui32 stationId = part.station();
        TString currentPart;

        Y_ENSURE(stationId || settlementId);

        if (stationId && settlementId) {
            const auto& settlement = raspDatabase.GetItemWithOuterId<TSettlementWrapper>(settlementId);
            const auto& station = raspDatabase.GetItemWithOuterId<TStationWrapper>(stationId);
            currentPart =
                TStringBuilder() << PartToText(settlement, lang) << " (" << PartToText(station, lang) << ")";
        } else if (stationId) {
            const auto& station = raspDatabase.GetItemWithOuterId<TStationWrapper>(stationId);
            currentPart = PartToText(station, lang);
        } else if (settlementId) {
            const auto& settlement = raspDatabase.GetItemWithOuterId<TSettlementWrapper>(settlementId);
            currentPart = PartToText(settlement, lang);
        }
        if (!isFirst)
            routeTitle += " \u2014 ";
        isFirst = false;
        routeTitle += currentPart;
    }

    if (titleCommon.is_ring()) {
        if (titleCommon.type() == TRThread::TRThreadTitle::MTA)
            routeTitle = translations.at("/ кольцевой")(routeTitle, lang);
        if (titleCommon.type() == TRThread::TRThreadTitle::SUBURBAN)
            routeTitle = translations.at("(кольцо)")(routeTitle, lang);
    }

    if (titleCommon.is_combined()) {
        if (titleCommon.type() == TRThread::TRThreadTitle::SUBURBAN)
            return translations.at("(согласованные поезда)")(routeTitle, lang);
        if (titleCommon.t_type() == TRThread::TRAIN) {
            if (titleCommon.title_parts().size() == 3) {
                return translations.at("(с пересадкой)")(routeTitle, lang);
            }
            return translations.at("(с пересадками)")(routeTitle, lang);
        }
    }

    return routeTitle;
}

TString NRasp::TSuburbanTitleGenerator::PartToText(const NRasp::TStationWrapper& pointWrapper, ELanguage lang) const {
    const auto& translations = TLanguageConfig::GetTranslations();
    auto title = TDefaultTitleGenerator::PartToText(pointWrapper, lang);
    return pointWrapper.IsAirport() ? translations.at("аэропорт")(title, lang) : title;
}
