#include "travel/rasp/route-search-api/datetime_helpers.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/timezone_conversion/convert.h>

#include <util/random/random.h>
#include <util/draft/datetime.h>
#include <util/datetime/parser.h>

Y_UNIT_TEST_SUITE(TestScheduleRangeProvider) {
    using namespace NDatetime;

    class TTestScheduleRangeProvider: public TScheduleRangeProvider {
        NDatetime::TSimpleTM GetNow() override {
            return NDatetime::CreateCivilTime(NDatetime::GetUtcTimeZone(), 2018, 1, 2, 10);
        }
    };

    TTimeZone pointTimeZone = GetTimeZone("Asia/Vladivostok"); // +10
    TTestScheduleRangeProvider provider;

    Y_UNIT_TEST(SearchDateIsEmpty) {
        // we should get a schedule between the local station time and the local station time + 30 days
        TMaybe<NDatetime::TSimpleTM> searchDate;
        auto [minDate, maxDate] = provider.GetRange(pointTimeZone, searchDate);

        UNIT_ASSERT_EQUAL(minDate, CreateCivilTime(pointTimeZone, 2018, 1, 2, 20));
        UNIT_ASSERT_EQUAL(maxDate, CreateCivilTime(pointTimeZone, 2018, 2, 1, 20));
    }
    Y_UNIT_TEST(SearchDateAndStationDateIsEqual) {
        // we should get a schedule between the local station time and the local station time + 1 days
        TMaybe<NDatetime::TSimpleTM> searchDate(CreateCivilTime(pointTimeZone, 2018, 1, 2));
        auto [minDate, maxDate] = provider.GetRange(pointTimeZone, searchDate);

        UNIT_ASSERT_EQUAL(minDate, CreateCivilTime(pointTimeZone, 2018, 1, 2, 20));
        UNIT_ASSERT_EQUAL(maxDate, CreateCivilTime(pointTimeZone, 2018, 1, 3, 0));
    }
    Y_UNIT_TEST(SearchDateIsFeatureDate) {
        // we should get a schedule between the search date and the search date + 1 days
        TMaybe<NDatetime::TSimpleTM> searchDate(CreateCivilTime(pointTimeZone, 2018, 1, 3));
        auto [minDate, maxDate] = provider.GetRange(pointTimeZone, searchDate);

        UNIT_ASSERT_EQUAL(minDate, CreateCivilTime(pointTimeZone, 2018, 1, 3, 0));
        UNIT_ASSERT_EQUAL(maxDate, CreateCivilTime(pointTimeZone, 2018, 1, 4, 0));
    }
    Y_UNIT_TEST(SearchDateIsPastDate) {
        // we should get schedule between the local station time and the local station time + 30 days
        TMaybe<NDatetime::TSimpleTM> searchDate(CreateCivilTime(pointTimeZone, 2017, 1, 1));
        auto [minDate, maxDate] = provider.GetRange(pointTimeZone, searchDate);

        UNIT_ASSERT_EQUAL(minDate, CreateCivilTime(pointTimeZone, 2018, 1, 2, 20));
        UNIT_ASSERT_EQUAL(maxDate, CreateCivilTime(pointTimeZone, 2018, 2, 1, 20));
    }
}

Y_UNIT_TEST_SUITE(TestParseCivilTime) {
    using namespace NDatetime;

    TTimeZone pointTimeZone = GetTimeZone("Asia/Vladivostok"); // +10

    Y_UNIT_TEST(Empty) {
        UNIT_ASSERT_EQUAL(ParseCivilTime(pointTimeZone, ""), TMaybe<TSimpleTM>());
    }
    Y_UNIT_TEST(WrongFormats) {
        TVector<TString> values{"wrong", "2017", "2017-", "2017-17", "2018-17-", "2018-19-20-"};

        for (auto& v : values) {
            UNIT_ASSERT_EQUAL(ParseCivilTime(pointTimeZone, v), TMaybe<TSimpleTM>());
        }
    }
    Y_UNIT_TEST(OkFormat) {
        TVector<std::pair<TString, TSimpleTM>> values{
            {"2017-01-02", CreateCivilTime(pointTimeZone, 2017, 1, 2)},
            {"2020-03-04", CreateCivilTime(pointTimeZone, 2020, 3, 4)}};

        for (auto& [v, expected] : values) {
            UNIT_ASSERT_EQUAL(ParseCivilTime(pointTimeZone, v), expected);
        }
    }
}

Y_UNIT_TEST_SUITE(TestAddMinutes) {
    using namespace NDatetime;

    Y_UNIT_TEST(AddPositive) {
        TSimpleTM dateTime(2018, 10, 1, 12, 30);
        UNIT_ASSERT_EQUAL(AddMinutes(dateTime, 60 + 30), TSimpleTM(2018, 10, 1, 14, 0));
    }

    Y_UNIT_TEST(AddZero) {
        TSimpleTM dateTime(2018, 10, 1, 12, 30);
        UNIT_ASSERT_EQUAL(AddMinutes(dateTime, 0), dateTime);
    }

    Y_UNIT_TEST(AddNegative) {
        TSimpleTM dateTime(2018, 10, 1, 12, 30);
        UNIT_ASSERT_EQUAL(AddMinutes(dateTime, -60 - 30), TSimpleTM(2018, 10, 1, 11, 0));
    }
}

Y_UNIT_TEST_SUITE(TestChangeTimeZone) {
    using namespace NDatetime;

    Y_UNIT_TEST(AddPositive) {
        TTimeZone msk = NDatetime::GetTimeZone("Europe/Moscow");
        TTimeZone vld = NDatetime::GetTimeZone("Asia/Vladivostok");
        TSimpleTM dateTime = CreateCivilTime(msk, 2018, 10, 1, 12, 30);

        UNIT_ASSERT_EQUAL(ChangeTimeZone(dateTime, msk, vld), CreateCivilTime(vld, 2018, 10, 1, 19, 30));
    }
}
