#include <travel/rasp/route-search-api/point_key.h>
#include <travel/rasp/route-search-api/wrappers.h>
#include <travel/rasp/route-search-api/limit_condition.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/vector.h>

Y_UNIT_TEST_SUITE(TestLimitCondition) {
    using namespace NRasp;

    TStation stationInFirstCity, stationInSecondCity, secondStationInFirstCity;

    TVector<TStationWrapper> stations = {
        TStationWrapper(stationInFirstCity, 1, {}, 1),
        TStationWrapper(stationInSecondCity, 2, {}, 2),
        TStationWrapper(secondStationInFirstCity, 3, {}, 1)};

    TPointKey settlementFrom(1, TPointKey::EPointKeyType::Settlement);
    TPointKey settlementTo(2, TPointKey::EPointKeyType::Settlement);

    TPointKey stationFrom(1, TPointKey::EPointKeyType::Station);
    TPointKey stationTo(2, TPointKey::EPointKeyType::Station);
    TPointKey stationToSame(3, TPointKey::EPointKeyType::Station);

    TTransportSet transportTypes = {
        TRThread::TRAIN,
        TRThread::PLANE};

    TTransportSet transportTypesWithBus = {
        TRThread::TRAIN,
        TRThread::PLANE,
        TRThread::BUS};

    Y_UNIT_TEST(FromSettlementToSettlement) {
        auto conditions = TLimitConditions(settlementFrom, settlementTo, transportTypes, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypes);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), TStation::NOT_IN_TABLO_ID);
    }

    Y_UNIT_TEST(FromCityToStationInOtherCity) {
        auto conditions = TLimitConditions(stationFrom, stationTo, transportTypes, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypes);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.ToMaxMajorityID(), TStation::STATION_ID);
    }

    Y_UNIT_TEST(FromStationToStationInSameCity) {
        auto conditions = TLimitConditions(stationFrom, stationToSame, transportTypes, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypes);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), TStation::STATION_ID);
    }

    Y_UNIT_TEST(FromStationToSameCity) {
        auto conditions = TLimitConditions(stationFrom, settlementFrom, transportTypes, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypes);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), TStation::STATION_ID);
    }

    Y_UNIT_TEST(FromSettlementToStationInSameCity) {
        auto conditions = TLimitConditions(settlementTo, stationTo, transportTypes, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypes);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.ToMaxMajorityID(), TStation::STATION_ID);
    }

    Y_UNIT_TEST(FromStationToOtherCity) {
        auto conditions = TLimitConditions(stationFrom, settlementTo, transportTypes, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypes);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), TStation::STATION_ID);
        UNIT_ASSERT_EQUAL(conditions.ToMaxMajorityID(), TStation::NOT_IN_TABLO_ID);
    }

    Y_UNIT_TEST(FromSettlementToStationInOtherCity) {
        auto conditions = TLimitConditions(settlementFrom, stationTo, transportTypes, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypes);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), TStation::NOT_IN_TABLO_ID);
        UNIT_ASSERT_EQUAL(conditions.ToMaxMajorityID(), TStation::STATION_ID);
    }

    Y_UNIT_TEST(FromSettlementToStationInSameCityWithBus) {
        auto conditions = TLimitConditions(settlementFrom, stationToSame, transportTypesWithBus, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), TTransportSet{TRThread::BUS});
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.ToMaxMajorityID(), TStation::STATION_ID);
    }

    Y_UNIT_TEST(FromStationToStationInSameCityWithBus) {
        auto conditions = TLimitConditions(stationFrom, stationToSame, transportTypesWithBus, stations);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), TTransportSet{TRThread::BUS});
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.ToMaxMajorityID(), TStation::STATION_ID);
    }

    Y_UNIT_TEST(TestExtended) {
        auto conditions = TLimitConditions(stationFrom, settlementFrom, transportTypesWithBus, stations, true);
        UNIT_ASSERT_EQUAL(conditions.TransportTypes(), transportTypesWithBus);
        UNIT_ASSERT_EQUAL(conditions.FromMaxMajorityID(), conditions.ToMaxMajorityID());
        UNIT_ASSERT_EQUAL(conditions.ToMaxMajorityID(), TStation::STATION_ID);
    }
}
