#include "travel/rasp/route-search-api/point_key.h"

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/string.h>
#include <util/random/random.h>

std::tuple<TString, NRasp::TPointKey> GenerateCorrectTest() {
    NRasp::TPointKey::EPointKeyType type = RandomNumber<bool>()
                                               ? NRasp::TPointKey::EPointKeyType::Settlement
                                               : NRasp::TPointKey::EPointKeyType::Station;
    auto id = RandomNumber<unsigned int>();
    TString prefix = (type == NRasp::TPointKey::EPointKeyType::Settlement) ? "c" : "s";

    return {TStringBuilder() << prefix << id, NRasp::TPointKey(id, type)};
};

Y_UNIT_TEST_SUITE(PointKeyTests) {
    using namespace NRasp;

    Y_UNIT_TEST(TestSettlement) {
        TPointKey pointKey(1, TPointKey::EPointKeyType::Settlement);
        UNIT_ASSERT(pointKey.IsSettlement());
        UNIT_ASSERT(!pointKey.IsStation());
        UNIT_ASSERT_EQUAL(pointKey.Id(), 1);
        UNIT_ASSERT_EQUAL(pointKey.Type(), TPointKey::EPointKeyType::Settlement);
    }

    Y_UNIT_TEST(TestStation) {
        TPointKey pointKey(2, TPointKey::EPointKeyType::Station);
        UNIT_ASSERT(pointKey.IsStation());
        UNIT_ASSERT(!pointKey.IsSettlement());
        UNIT_ASSERT_EQUAL(pointKey.Id(), 2);
        UNIT_ASSERT_EQUAL(pointKey.Type(), TPointKey::EPointKeyType::Station);
    }

    Y_UNIT_TEST(TestCompare) {
        TPointKey pointKey(1, TPointKey::EPointKeyType::Settlement);
        TPointKey otherPointKey(1, TPointKey::EPointKeyType::Settlement);
        UNIT_ASSERT_EQUAL(pointKey, otherPointKey);

        otherPointKey = TPointKey(2, TPointKey::EPointKeyType::Settlement);
        UNIT_ASSERT_UNEQUAL(pointKey, otherPointKey);

        otherPointKey = TPointKey(1, TPointKey::EPointKeyType::Station);
        UNIT_ASSERT_UNEQUAL(pointKey, otherPointKey);

        otherPointKey = TPointKey(2, TPointKey::EPointKeyType::Station);
        UNIT_ASSERT_UNEQUAL(pointKey, otherPointKey);
    }

    Y_UNIT_TEST(TestParse) {
        TVector<std::tuple<TString, TPointKey>> tests = {
            {"s100", TPointKey(100, TPointKey::EPointKeyType::Station)},
            {"c1", TPointKey(1, TPointKey::EPointKeyType::Settlement)}};

        auto testCount = 100000;

        for (int i = 0; i < testCount; i++)
            tests.push_back(GenerateCorrectTest());

        for (auto& [toParse, expected] : tests) {
            TPointKey pointKey;

            UNIT_ASSERT_C(pointKey.TryParseFromString(toParse),
                          TStringBuilder() << "Can't parse " << toParse);
            UNIT_ASSERT_EQUAL(pointKey, expected);
        }
    }

    Y_UNIT_TEST(TestNotParse) {
        static TVector<TString> toParse = {
            "s-100",
            "c",
            "123",
            "x12",
            "",
            "c0",
            nullptr,
        };

        for (auto& testString : toParse) {
            TPointKey pointKey;
            UNIT_ASSERT_C(
                !pointKey.TryParseFromString(testString),
                TStringBuilder() << "Can parse string " << testString);
        }
    }

    Y_UNIT_TEST(TestPrintdddd) {
        UNIT_ASSERT_EQUAL(TPointKey(6, TPointKey::EPointKeyType::Station).ToString(), "s6");
        UNIT_ASSERT_EQUAL(TPointKey(666, TPointKey::EPointKeyType::Settlement).ToString(), "c666");
    }
}
