#include "travel/rasp/route-search-api/query_reader.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/timezone_conversion/civil.h>
#include <util/generic/string.h>
#include <util/random/random.h>

Y_UNIT_TEST_SUITE(QueryReaderTests) {
    using namespace NRasp;

    TRaspDatabase MakeDatabase() {
        TSettlement msk;
        msk.set_id(100);
        msk.set_time_zone(1);

        TSettlement ekb;
        ekb.set_id(12);
        ekb.set_time_zone(1);

        TStation mskStation;
        mskStation.set_id(10);
        mskStation.set_time_zone(1);

        TStation ekbStation;
        ekbStation.set_id(11);
        ekbStation.set_time_zone(1);

        THashMap<int, NDatetime::TTimeZone> idToTimezone;
        TVector<TRThread> rthreads;
        TVector<TThreadStation> rtstations;
        TVector<TSettlement> settlements;
        TVector<TStation> stations;
        TVector<TStation2Settlement> stationToSettlements;
        TVector<TThreadTariff> threadTariffs;
        TVector<TSupplier> suppliers;
        THashMap<object_id_t, TString> uidById;
        TVector<TCurrencyProto> currencies;

        settlements.push_back(msk);
        settlements.push_back(ekb);
        stations.push_back(mskStation);
        stations.push_back(ekbStation);

        idToTimezone[1] = NDatetime::GetTimeZone("Europe/Moscow");

        NRasp::TRaspDatabase database(idToTimezone, rthreads, rtstations, settlements, stations, stationToSettlements, threadTariffs, suppliers, uidById, currencies);
        return database;
    }

    Y_UNIT_TEST(SimpleTest) {
        auto reader = TDirectionQueryReader();

        auto database = MakeDatabase();
        NRasp::TWrappedRaspDatabase databaseWrapper{database};

        TStringStream ss("c100 s10 2018-10-13 ru test_reqid");
        TMaybe<TDirectionQuery> query = reader.ReadQuery(ss, databaseWrapper);
        UNIT_ASSERT(query.Defined());
        UNIT_ASSERT_EQUAL(query.GetRef().FromPointKey, TPointKey(100, TPointKey::EPointKeyType::Settlement));
        UNIT_ASSERT_EQUAL(query.GetRef().ToPointKey, TPointKey(10, TPointKey::EPointKeyType::Station));
        auto minDate = query.GetRef().SearchDate;

        UNIT_ASSERT_EQUAL(minDate->MDay, 13);
        UNIT_ASSERT_EQUAL(minDate->RealMonth(), 10);
        UNIT_ASSERT_EQUAL(minDate->RealYear(), 2018);

        UNIT_ASSERT_EQUAL(query.GetRef().Tld, "ru");
        UNIT_ASSERT_EQUAL(query.GetRef().MainReqid, "test_reqid");

        query = reader.ReadQuery(ss, databaseWrapper);
        UNIT_ASSERT(query.Empty());
    }

    Y_UNIT_TEST(TestManyQueriesInStream) {
        NRasp::TRaspDatabase database = MakeDatabase();
        NRasp::TWrappedRaspDatabase databaseWrapper{database};
        auto reader = TDirectionQueryReader();
        TStringStream ss("c100 s10 2018-10-13 ru test_reqid\ns11 c12 2017-07-13 ua test_reqid");

        TMaybe<TDirectionQuery> query = reader.ReadQuery(ss, databaseWrapper);
        UNIT_ASSERT(query.Defined());
        UNIT_ASSERT_EQUAL(query.GetRef().FromPointKey, TPointKey(100, TPointKey::EPointKeyType::Settlement));
        UNIT_ASSERT_EQUAL(query.GetRef().ToPointKey, TPointKey(10, TPointKey::EPointKeyType::Station));
        auto minDate = query.GetRef().SearchDate;

        UNIT_ASSERT_EQUAL(minDate->MDay, 13);
        UNIT_ASSERT_EQUAL(minDate->RealMonth(), 10);
        UNIT_ASSERT_EQUAL(minDate->RealYear(), 2018);

        UNIT_ASSERT_EQUAL(query.GetRef().Tld, "ru");
        UNIT_ASSERT_EQUAL(query.GetRef().MainReqid, "test_reqid");

        query = reader.ReadQuery(ss, databaseWrapper);
        UNIT_ASSERT_EQUAL(query.GetRef().FromPointKey, TPointKey(11, TPointKey::EPointKeyType::Station));
        UNIT_ASSERT_EQUAL(query.GetRef().ToPointKey, TPointKey(12, TPointKey::EPointKeyType::Settlement));

        minDate = query.GetRef().SearchDate;

        UNIT_ASSERT_EQUAL(minDate->MDay, 13);
        UNIT_ASSERT_EQUAL(minDate->RealMonth(), 7);
        UNIT_ASSERT_EQUAL(minDate->RealYear(), 2017);

        UNIT_ASSERT_EQUAL(query.GetRef().Tld, "ua");

        query = reader.ReadQuery(ss, databaseWrapper);
        UNIT_ASSERT(query.Empty());
    }
}
