#include "test_helpers.h"

#include <travel/rasp/route-search-api/rasp_database.h>
#include <travel/rasp/route-search-api/helpers.h>
#include <travel/rasp/route-search-api/point_key.h>
#include <travel/rasp/route-search-api/proto/rasp_precache.pb.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/random/random.h>

Y_UNIT_TEST_SUITE(RaspDatabaseTests) {
    using namespace NRasp;

    Y_UNIT_TEST(TestInitRaspDatabase) {
        auto raspDatabase = GetDatabase();
        UNIT_ASSERT_EQUAL(raspDatabase.GetItems<TSettlement>(), GetSettlements());
        UNIT_ASSERT_EQUAL(raspDatabase.GetItems<TStation>(), GetStations());
        UNIT_ASSERT_EQUAL(raspDatabase.GetItems<TRThread>(), GetRThreads());
        UNIT_ASSERT_EQUAL(raspDatabase.GetItems<TThreadStation>(), GetRTStations());
        UNIT_ASSERT_EQUAL(raspDatabase.GetItems<TStation2Settlement>(), GetStationToSettlements());
        UNIT_ASSERT_EQUAL(raspDatabase.GetTimezoneById(1).name(), "Europe/Moscow");
        UNIT_ASSERT_EQUAL(raspDatabase.GetTimezoneById(2).name(), "Asia/Yekaterinburg");
    }

    Y_UNIT_TEST(TestWrappedTimezones) {
        auto database = GetDatabase();
        auto raspDatabase = TWrappedRaspDatabase(database);
        UNIT_ASSERT_EQUAL(raspDatabase.GetTimezoneById(1).name(), "Europe/Moscow");
        UNIT_ASSERT_EQUAL(raspDatabase.GetTimezoneById(2).name(), "Asia/Yekaterinburg");
    }

    Y_UNIT_TEST(TestWrappedSettlemnts) {
        auto database = GetDatabase();
        auto raspDatabase = TWrappedRaspDatabase(database);
        auto settlements = GetSettlements();

        for (size_t i = 0; i < settlements.size(); i++) {
            UNIT_ASSERT_EQUAL(raspDatabase.GetInnerId<TSettlement>(settlements[i].id()), i + 1);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TSettlementWrapper>(i + 1).Item(), settlements[i]);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TSettlementWrapper>(i + 1).Id(), i + 1);
        }
    }

    Y_UNIT_TEST(TestWrappedStations) {
        auto database = GetDatabase();
        auto raspDatabase = TWrappedRaspDatabase(database);
        auto stations = GetStations();

        for (size_t i = 0; i < stations.size(); i++) {
            UNIT_ASSERT_EQUAL(raspDatabase.GetInnerId<TStation>(stations[i].id()), i + 1);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TStationWrapper>(i + 1).Item(), stations[i]);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TStationWrapper>(i + 1).Id(), i + 1);
        }
    }

    Y_UNIT_TEST(TestWrappedThreads) {
        auto database = GetDatabase();
        auto raspDatabase = TWrappedRaspDatabase(database);
        auto threads = GetRThreads();

        for (size_t i = 0; i < threads.size(); i++) {
            UNIT_ASSERT_EQUAL(raspDatabase.GetInnerId<TRThread>(threads[i].id()), i + 1);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TRThreadWrapper>(i + 1).Item(), threads[i]);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TRThreadWrapper>(i + 1).Id(), i + 1);
        }
    }

    Y_UNIT_TEST(TestWrappedRTStations) {
        auto database = GetDatabase();
        auto raspDatabase = TWrappedRaspDatabase(database);
        auto rtstations = GetRTStations();

        for (size_t i = 0; i < rtstations.size(); i++) {
            UNIT_ASSERT_EQUAL(raspDatabase.GetInnerId<TThreadStation>(rtstations[i].id()), i + 1);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TThreadStationWrapper>(i + 1).Item(), rtstations[i]);
            UNIT_ASSERT_EQUAL(raspDatabase.GetItemWithId<TThreadStationWrapper>(i + 1).Id(), i + 1);
        }
    }

    Y_UNIT_TEST(TestObjectReader) {
        TSettlementPack pack;
        for (const auto& settlement : GetSettlements()) {
            *pack.add_items() = settlement;
        }
        TStringStream stringStream(pack.SerializeAsString());
        auto reader = TObjectStreamReader<TSettlementPack>();
        auto settlements = reader.Read(stringStream);
        UNIT_ASSERT_EQUAL(settlements, GetSettlements());
    }

    Y_UNIT_TEST(TestTimezoneReader) {
        TRaspPrecache timezones;
        timezones.mutable_timezone_ids()->insert({"Europe/Minsk", 0});
        timezones.mutable_timezone_ids()->insert({"Europe/Moscow", 1});
        timezones.mutable_timezone_ids()->insert({"Asia/Yekaterinburg", 2});
        TStringStream stringStream(timezones.SerializeAsString());
        auto reader = TTimezonesStreamReader();
        UNIT_ASSERT_EQUAL(reader.Read(stringStream), GetTimezones());
    }

    Y_UNIT_TEST(TestPointKeyToInnerId) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);

        auto settlements = GetSettlements();
        for (ui32 i = 0; i < settlements.size(); i++) {
            TPointKey pointKey(settlements[i].id(), TPointKey::EPointKeyType::Settlement);
            UNIT_ASSERT_EQUAL(pointKey.ToInnerIds(wrapperDatabase),
                              TPointKey(i + 1, TPointKey::EPointKeyType::Settlement));
        }

        auto stations = GetStations();
        for (ui32 i = 0; i < stations.size(); i++) {
            TPointKey pointKey(stations[i].id(), TPointKey::EPointKeyType::Station);
            UNIT_ASSERT_EQUAL(pointKey.ToInnerIds(wrapperDatabase),
                              TPointKey(i + 1, TPointKey::EPointKeyType::Station));
        }
    }
}
