#include "test_helpers.h"
#include <travel/rasp/route-search-api/rasp_search_index.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/random/random.h>

Y_UNIT_TEST_SUITE(RaspSearchIndexTests) {
    using namespace NRasp;

    Y_UNIT_TEST(TestFetchStationSettlement) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);

        auto answer = FetchStationSettlements(
            wrapperDatabase.GetItems<TStationWrapper>(),
            wrapperDatabase.GetItems<TStationToSettlementWrapper>());

        THashMap<object_id_t, TVector<object_id_t>> expected = {
            {1, {1, 2}},
            {2, {1}},
            {3, {2}},
            {4, {3}},
            {5, {4}},
            {6, {5}},
            {7, {2}},
            {8, {2}},
        };

        for (object_id_t id = 9; id < 15; id++) {
            expected[id].push_back(6);
        }

        for (object_id_t id = 15; id < 21; id++) {
            expected[id].push_back(7);
        }

        UNIT_ASSERT_EQUAL(answer, expected);
    }

    Y_UNIT_TEST(TestFetchThreadStations) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);

        auto answer = FetchThreadStations(wrapperDatabase.GetItems<TThreadStationWrapper>());

        THashMap<object_id_t, TVector<object_id_t>> expected = {
            {1, {1, 3}},
            {2, {2, 4}},
            {3, {5, 7, 13}},
            {4, {6, 8}},
            {5, {9, 11}},
            {6, {10, 12}},
            {7, {14, 15}},
            {8, {16, 17, 18, 19}},
            {9, {20, 21, 22, 23, 24, 25}},
            {10, {26, 27}},
            {11, {28, 29}},
            {12, {30, 31, 32}}};

        UNIT_ASSERT_EQUAL(answer, expected);
    }

    Y_UNIT_TEST(TestInit) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);
        TTransportSet transportTypes = {
            TRThread::TRAIN,
            TRThread::SUBURBAN};
        auto searchIndex = TRaspSearchIndex(wrapperDatabase, transportTypes);
        UNIT_ASSERT_EQUAL(searchIndex.GetTransportTypes(), transportTypes);
    }

    Y_UNIT_TEST(TestStationPointStops) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);
        TTransportSet transportTypes = {
            TRThread::TRAIN,
            TRThread::SUBURBAN,
            TRThread::PLANE,
            TRThread::BUS,
        };
        auto searchIndex = TRaspSearchIndex(wrapperDatabase, transportTypes);
        auto rtstations = GetRTStations();

        {
            const TPointStops& pointStops = searchIndex.GetPointStops({1, TPointKey::EPointKeyType::Station});
            UNIT_ASSERT_EQUAL(pointStops.GetKeys(), TVector<object_id_t>({1, 2}));

            UNIT_ASSERT_EQUAL(pointStops.at(1).ysize(), 1);
            UNIT_ASSERT_EQUAL(pointStops.at(1)[0]->Item(), rtstations[0]);

            UNIT_ASSERT_EQUAL(pointStops.at(2).ysize(), 1);
            UNIT_ASSERT_EQUAL(pointStops.at(2)[0]->Item(), rtstations[1]);
        }

        {
            const TPointStops& pointStops = searchIndex.GetPointStops({2, TPointKey::EPointKeyType::Station});
            UNIT_ASSERT_EQUAL(pointStops.GetKeys(), TVector<object_id_t>({1, 2}));

            UNIT_ASSERT_EQUAL(pointStops.at(1).ysize(), 1);
            UNIT_ASSERT_EQUAL(pointStops.at(1)[0]->Item(), rtstations[2]);

            UNIT_ASSERT_EQUAL(pointStops.at(2).ysize(), 1);
            UNIT_ASSERT_EQUAL(pointStops.at(2)[0]->Item(), rtstations[3]);
        }
    }

    Y_UNIT_TEST(SettlementPointStops) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);
        TTransportSet transportTypes = {
            TRThread::TRAIN,
            TRThread::SUBURBAN,
            TRThread::PLANE,
            TRThread::BUS,
        };
        auto searchIndex = TRaspSearchIndex(wrapperDatabase, transportTypes);
        auto rtstations = GetRTStations();

        const TPointStops& pointStops = searchIndex.GetPointStops({6, TPointKey::EPointKeyType::Settlement});

        UNIT_ASSERT_EQUAL(pointStops.GetKeys(), TVector<object_id_t>({7, 8, 9}));

        UNIT_ASSERT_EQUAL(pointStops.at(7).ysize(), 1);
        UNIT_ASSERT_EQUAL(pointStops.at(8).ysize(), 2);
        UNIT_ASSERT_EQUAL(pointStops.at(9).ysize(), 3);

        UNIT_ASSERT_EQUAL(pointStops.at(7)[0]->Item(), rtstations[13]);

        UNIT_ASSERT_EQUAL(pointStops.at(8)[0]->Item(), rtstations[15]);
        UNIT_ASSERT_EQUAL(pointStops.at(8)[1]->Item(), rtstations[16]);

        UNIT_ASSERT_EQUAL(pointStops.at(9)[0]->Item(), rtstations[19]);
        UNIT_ASSERT_EQUAL(pointStops.at(9)[1]->Item(), rtstations[20]);
        UNIT_ASSERT_EQUAL(pointStops.at(9)[2]->Item(), rtstations[21]);
    }
}
