#include "travel/rasp/route-search-api/segment_finders.h"
#include "travel/rasp/route-search-api/ut/test_helpers.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/random/random.h>

ui32 ParseTime(const TString& time) {
    TString hoursString, minutesString;
    Split(time, ":", hoursString, minutesString);
    int hours, minutes;

    TryIntFromString<10>(hoursString, hours);
    TryIntFromString<10>(minutesString, minutes);

    return hours * 60 + minutes;
}

NDatetime::TSimpleTM ParseDate(const TString& date) {
    ui32 year, month, day;
    Split(date, '-', year, month, day);
    return NDatetime::TSimpleTM(year, month, day);
}

NDatetime::TSimpleTM ParseSimpleDate(const TString& date) {
    TIso8601DateTimeParser parser;
    parser.ParsePart(date.c_str(), date.size());
    auto fields = parser.GetDateTimeFields();
    return NDatetime::TSimpleTM::New(fields.ToTimeT({}));
}

Y_UNIT_TEST_SUITE(CalculateThreadStart) {
    class TTestData {
        NDatetime::TTimeZone Timezone;
        NDatetime::TSimpleTM MinEventTM;
        ui32 ThreadStartTime;
        ui32 EventOffset;
        NDatetime::TSimpleTM Expected;

    public:
        TTestData(const TString& timezone, const TString& minEventTM,
                  const TString& threadStartTime, const ui32 eventOffset, const TString& expected)
            : Timezone(NDatetime::GetTimeZone(timezone))
            , MinEventTM(ParseSimpleDate(minEventTM))
            , ThreadStartTime(ParseTime(threadStartTime))
            , EventOffset(eventOffset)
            , Expected(ParseSimpleDate(expected))
        {
        }

        void Run() const {
            auto result = NRasp::CalculateThreadStart(Timezone, MinEventTM, ThreadStartTime, EventOffset);
            UNIT_ASSERT_EQUAL_C(result, Expected,
                                TStringBuilder() << Endl << result.ToString() << " != " << Expected.ToString());
        }
    };

    Y_UNIT_TEST(Today) {
        TTestData("Europe/Moscow", "2018-08-10T00:00", "06:00", 0, "2018-08-10T06:00+03:00").Run();
    }

    Y_UNIT_TEST(Tomorrow) {
        TTestData("Europe/Moscow", "2018-08-10T12:00", "06:00", 0, "2018-08-11T06:00+03:00").Run();
    }

    Y_UNIT_TEST(Yesterday) {
        TTestData("Europe/Moscow", "2018-08-10T00:00", "06:00", 24 * 60, "2018-08-09T06:00+03:00").Run();
    }

    Y_UNIT_TEST(YesterdayInOtherMonth) {
        TTestData("Europe/Moscow", "2018-08-01T00:00", "06:00", 24 * 60, "2018-07-31T06:00+03:00").Run();
        TTestData("Europe/Moscow", "2018-07-01T00:00", "06:00", 24 * 60, "2018-06-30T06:00+03:00").Run();
    }

    Y_UNIT_TEST(YesterdayInOtherYear) {
        TTestData("Europe/Moscow", "2018-01-01T00:00", "06:00", 24 * 60, "2017-12-31T06:00+03:00").Run();
    }

    Y_UNIT_TEST(LeapYear) {
        TTestData("Europe/Moscow", "2018-03-01T00:00", "06:00", 24 * 60, "2018-02-28T06:00+03:00").Run();
        TTestData("Europe/Moscow", "2016-03-01T00:00", "06:00", 24 * 60, "2016-02-29T06:00+03:00").Run();
    }
}

Y_UNIT_TEST_SUITE(SegmentsFinder) {
    using namespace NRasp;

    Y_UNIT_TEST(CommonSegmentFinder) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);
        TTransportSet transportTypes = {
            TRThread::TRAIN,
            TRThread::SUBURBAN,
            TRThread::PLANE,
            TRThread::BUS,
        };
        auto searchIndex = TRaspSearchIndex(wrapperDatabase, transportTypes);

        auto threads = GetRThreads();
        auto rtstations = GetRTStations();

        auto fromPointKey = TPointKey(16, TPointKey::EPointKeyType::Settlement);
        auto toPointKey = TPointKey(17, TPointKey::EPointKeyType::Settlement);

        fromPointKey = fromPointKey.ToInnerIds(wrapperDatabase);
        toPointKey = toPointKey.ToInnerIds(wrapperDatabase);

        auto majorityLimit = TLimitConditions(
            fromPointKey,
            toPointKey,
            searchIndex.GetTransportTypes(),
            searchIndex.GetItems<TStationWrapper>());

        TCommonSegmentFinder findSegments;

        auto result = findSegments.Find(searchIndex, fromPointKey, toPointKey, majorityLimit);
        UNIT_ASSERT_EQUAL(result.ysize(), 3);

        UNIT_ASSERT_EQUAL(result[0]->Thread().Item(), threads[6]);
        UNIT_ASSERT_EQUAL(result[1]->Thread().Item(), threads[7]);
        UNIT_ASSERT_EQUAL(result[2]->Thread().Item(), threads[8]);

        UNIT_ASSERT_EQUAL(result[0]->Departure().Item(), rtstations[13]);
        UNIT_ASSERT_EQUAL(result[1]->Departure().Item(), rtstations[16]);
        UNIT_ASSERT_EQUAL(result[2]->Departure().Item(), rtstations[19]);

        UNIT_ASSERT_EQUAL(result[0]->Arrival().Item(), rtstations[14]);
        UNIT_ASSERT_EQUAL(result[1]->Arrival().Item(), rtstations[17]);
        UNIT_ASSERT_EQUAL(result[2]->Arrival().Item(), rtstations[23]);
    }

    Y_UNIT_TEST(FilterThroughTrain) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);
        TTransportSet transportTypes = {
            TRThread::TRAIN,
            TRThread::SUBURBAN,
            TRThread::PLANE,
            TRThread::BUS,
        };
        auto searchIndex = TRaspSearchIndex(wrapperDatabase, transportTypes);

        auto threads = GetRThreads();
        auto rtstations = GetRTStations();

        auto fromPointKey = TPointKey(25, TPointKey::EPointKeyType::Station);
        auto toPointKey = TPointKey(30, TPointKey::EPointKeyType::Station);

        fromPointKey = fromPointKey.ToInnerIds(wrapperDatabase);
        toPointKey = toPointKey.ToInnerIds(wrapperDatabase);

        auto majorityLimit = TLimitConditions(
            fromPointKey,
            toPointKey,
            searchIndex.GetTransportTypes(),
            searchIndex.GetItems<TStationWrapper>());

        TCommonSegmentFinder findSegments;
        TThroughTrainsFilter filterThroughTrains;

        auto commonSegments = findSegments.Find(searchIndex, fromPointKey, toPointKey, majorityLimit);
        auto result = filterThroughTrains.Filter(searchIndex, std::move(commonSegments));

        UNIT_ASSERT_EQUAL(result.ysize(), 3);

        SortBy(result, [](const auto& x) { return x->Thread().Id(); });

        UNIT_ASSERT_EQUAL(result[0]->Thread().Item(), threads[2]);
        UNIT_ASSERT_EQUAL(result[1]->Thread().Item(), threads[3]);
        UNIT_ASSERT_EQUAL(result[2]->Thread().Item(), threads[10]);
    }

    Y_UNIT_TEST(DateFilter) {
        auto database = GetDatabase();
        auto wrapperDatabase = TWrappedRaspDatabase(database);
        TTransportSet transportTypes = {
            TRThread::TRAIN,
            TRThread::SUBURBAN,
            TRThread::PLANE,
            TRThread::BUS,
        };
        auto searchIndex = TRaspSearchIndex(wrapperDatabase, transportTypes);

        auto threads = GetRThreads();
        auto rtstations = GetRTStations();

        auto fromPointKey = TPointKey(40, TPointKey::EPointKeyType::Station);
        auto toPointKey = TPointKey(35, TPointKey::EPointKeyType::Station);

        fromPointKey = fromPointKey.ToInnerIds(wrapperDatabase);
        toPointKey = toPointKey.ToInnerIds(wrapperDatabase);

        auto majorityLimit = TLimitConditions(
            fromPointKey,
            toPointKey,
            searchIndex.GetTransportTypes(),
            searchIndex.GetItems<TStationWrapper>());

        TCommonSegmentFinder findSegments;
        TDateSegmentFilter dateSegmentFilter;

        auto commonSegments = findSegments.Find(searchIndex, fromPointKey, toPointKey, majorityLimit);

        UNIT_ASSERT_EQUAL(commonSegments.ysize(), 1);
        UNIT_ASSERT_EQUAL(commonSegments[0]->Thread().Item(), threads[11]);

        auto result = dateSegmentFilter.Filter(
            commonSegments, ParseDate("2018-08-15"), ParseDate("2018-08-16"),
            TCommonSegmentFinder::EEventType::Departure, false);

        UNIT_ASSERT_EQUAL(result.ysize(), 2);

        for (int i = 0; i < 2; i++) {
            UNIT_ASSERT_EQUAL(result[i]->Thread(), threads[11]);
            UNIT_ASSERT_EQUAL(result[i]->Departure(), rtstations[30]);
            UNIT_ASSERT_EQUAL(result[i]->Arrival(), rtstations[31]);
        }

        UNIT_ASSERT_EQUAL(result[0]->DepartureDt(), TInstant::ParseIso8601("2018-08-15T22:10"));
        UNIT_ASSERT_EQUAL(result[1]->DepartureDt(), TInstant::ParseIso8601("2018-08-16T22:10"));
    }
}
