#include "test_helpers.h"

#include <travel/rasp/route-search-api/rasp_database.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/langs/langs.h>
#include <util/random/random.h>
#include <library/cpp/resource/resource.h>

using namespace NRasp;

Y_UNIT_TEST_SUITE(SettlementGetTitle) {
    void TestSettlement(TSettlement & settlement, const THashMap<ELanguage, TString>& titles) {
        TSettlementWrapper settlementWrapper(settlement, 0);

        for (auto&& [lang, name] : titles) {
            UNIT_ASSERT_EQUAL_C(settlementWrapper.Title(lang), name,
                                TStringBuilder() << "Incorrect test for language " << FullNameByLanguage(lang)
                                                 << " : " << settlementWrapper.Title(lang) << " != " << name);
        }
    }

    Y_UNIT_TEST(WithoutTranslations) {
        TSettlement s;
        TestSettlement(s, {{LANG_RUS, ""},
                           {LANG_UKR, ""},
                           {LANG_TUR, ""}});
    }

    Y_UNIT_TEST(WithOnlyRuTranslations) {
        TSettlement s;
        s.set_title_ru("MOSCOW");
        TestSettlement(s, {{LANG_RUS, "MOSCOW"},
                           {LANG_UKR, "MOSCOW"},
                           {LANG_TUR, ""}});
    }

    Y_UNIT_TEST(WithOnlyUkTranslations) {
        TSettlement s;
        s.set_title_uk("MOSCOW");
        TestSettlement(s, {{LANG_RUS, ""},
                           {LANG_UKR, "MOSCOW"},
                           {LANG_TUR, ""}});
    }

    Y_UNIT_TEST(WithOnlyTrTranslations) {
        TSettlement s;
        s.set_title_tr("MOSCOW");
        TestSettlement(s, {{LANG_RUS, ""},
                           {LANG_UKR, ""},
                           {LANG_TUR, "MOSCOW"}});
    }

    Y_UNIT_TEST(WithOnlyCommonTranslations) {
        TSettlement s;
        s.set_title("MOSCOW");
        TestSettlement(s, {{LANG_RUS, "MOSCOW"},
                           {LANG_UKR, "MOSCOW"},
                           {LANG_TUR, "MOSCOW"}});
    }

    Y_UNIT_TEST(WithAllTranslations) {
        TSettlement s;
        s.set_title("MOSCOW_COMMON");
        s.set_title_ru("MOSCOW_RU");
        s.set_title_uk("MOSCOW_UK");
        s.set_title_tr("MOSCOW_TR");
        TestSettlement(s, {{LANG_RUS, "MOSCOW_RU"},
                           {LANG_UKR, "MOSCOW_UK"},
                           {LANG_TUR, "MOSCOW_TR"}});
    }
}

Y_UNIT_TEST_SUITE(Stations) {
    void TestStation(TStation & station, const THashMap<ELanguage, TString>& titles) {
        TSettlement settlement;
        TStationWrapper stationWrapper(station, 0, settlement, 0);

        for (auto&& [lang, name] : titles) {
            UNIT_ASSERT_EQUAL_C(stationWrapper.Title(lang), name,
                                TStringBuilder() << "Incorrect test for language " << FullNameByLanguage(lang)
                                                 << " : " << stationWrapper.Title(lang) << " != " << name);
        }
    }

    Y_UNIT_TEST(WithoutTranslations) {
        TStation s;
        TestStation(s, {{LANG_RUS, ""},
                        {LANG_UKR, ""},
                        {LANG_TUR, ""}});
    }

    Y_UNIT_TEST(WithOnlyRuTranslations) {
        TStation s;
        s.set_title_ru("KOLTSOVO");
        TestStation(s, {{LANG_RUS, "KOLTSOVO"},
                        {LANG_UKR, "KOLTSOVO"},
                        {LANG_TUR, ""}});
    }

    Y_UNIT_TEST(WithOnlyUkTranslations) {
        TStation s;
        s.set_title_uk("KOLTSOVO");
        TestStation(s, {{LANG_RUS, ""},
                        {LANG_UKR, "KOLTSOVO"},
                        {LANG_TUR, ""}});
    }

    Y_UNIT_TEST(WithOnlyTrTranslations) {
        TStation s;
        s.set_title_tr("KOLTSOVO");
        TestStation(s, {{LANG_RUS, ""},
                        {LANG_UKR, ""},
                        {LANG_TUR, "KOLTSOVO"}});
    }

    Y_UNIT_TEST(WithOnlyCommonTranslations) {
        TStation s;
        s.set_title("KOLTSOVO");
        TestStation(s, {{LANG_RUS, "KOLTSOVO"},
                        {LANG_UKR, "KOLTSOVO"},
                        {LANG_TUR, "KOLTSOVO"}});
    }

    Y_UNIT_TEST(WithAllTranslations) {
        TStation s;
        s.set_title("KOLTSOVO_COMMON");
        s.set_title_ru("KOLTSOVO_RU");
        s.set_title_uk("KOLTSOVO_UK");
        s.set_title_tr("KOLTSOVO_TR");
        TestStation(s, {{LANG_RUS, "KOLTSOVO_RU"},
                        {LANG_UKR, "KOLTSOVO_UK"},
                        {LANG_TUR, "KOLTSOVO_TR"}});
    }
}
