#include "wrappers.h"
#include "rasp_database.h"
#include "title_generator.h"
#include "config.h"

#include <util/generic/algorithm.h>
#include <library/cpp/timezone_conversion/convert.h>

using namespace NRasp;

static constexpr int BitsInMonth = 31;

template <class T>
TInnerIdWrapper<T>::TInnerIdWrapper(const T& item, const TWrappedRaspDatabase& info)
    : TWrapper<T>(item)
{
    InnerId = info.GetInnerId<T>(item.id());
}

TSettlementWrapper::TSettlementWrapper(const TSettlement& item, const TWrappedRaspDatabase& info)
    : TInnerIdWrapper<TSettlement>(item, info)
{
    Timezone_ = info.GetTimezoneById(item.time_zone());
}

TString TSettlementWrapper::PopularTitle(ELanguage language) const {
    switch (language) {
        case LANG_RUS:
            return GetFirstNotEmptyOrDefault({Item().title_ru(), Item().title()});
        case LANG_TUR:
            return GetFirstNotEmptyOrDefault({Item().title_tr(), Item().title_en(), Item().title()});
        case LANG_UKR:
            return GetFirstNotEmptyOrDefault({Item().title_uk(), Item().title_ru(), Item().title()});
        default:
            ythrow TUnknownLanguageException() << "Unknown language: " << FullNameByLanguage(language);
    }
}

TString TStationWrapper::Title(ELanguage language) const {
    switch (language) {
        case LANG_RUS:
            return GetFirstNotEmptyOrDefault({Item().title_ru(), Item().title()});
        case LANG_TUR:
            return GetFirstNotEmptyOrDefault({Item().title_tr(), Item().title_en(), Item().title()});
        case LANG_UKR:
            return GetFirstNotEmptyOrDefault({Item().title_uk(), Item().title_ru(), Item().title()});
        default:
            ythrow TUnknownLanguageException() << "Unknown language: " << FullNameByLanguage(language);
    }
}

TString TStationWrapper::PopularTitle(ELanguage language) const {
    switch (language) {
        case LANG_RUS:
            return GetFirstNotEmptyOrDefault({Item().popular_title_ru(), Item().popular_title(),
                                              Item().title_ru(), Item().title()});
        case LANG_TUR:
            return GetFirstNotEmptyOrDefault({Item().popular_title_tr(), Item().title_tr(),
                                              Item().popular_title_en(), Item().title_en(),
                                              Item().popular_title(), Item().title()});
        case LANG_UKR:
            return GetFirstNotEmptyOrDefault({Item().popular_title_uk(), Item().title_uk(),
                                              Item().popular_title_ru(), Item().popular_title(),
                                              Item().title_ru(), Item().title()});
        default:
            ythrow TUnknownLanguageException() << "Unknown language: " << FullNameByLanguage(language);
    }
}

TSettlementWrapper::TSettlementWrapper(const TSettlement& item, object_id_t id,
                                       const NDatetime::TTimeZone& timezone)
    : TInnerIdWrapper<TSettlement>(item, id)
    , Timezone_(timezone)
{
}

TRThreadWrapper::TRThreadWrapper(const TRThread& item, const TWrappedRaspDatabase& info)
    : TInnerIdWrapper<TRThread>(item, info)
    , Uid_(item.uid())
{
}

TRThreadWrapper::TRThreadWrapper(const TRThread& item, object_id_t pk, const TString& uid)
    : TInnerIdWrapper<TRThread>(item, pk)
    , Uid_(uid)
{
}

TString TSettlementWrapper::Title(ELanguage language) const {
    return PopularTitle(language);
}

const NDatetime::TTimeZone& TSettlementWrapper::Timezone() const {
    return Timezone_;
}

TRThread::ETypeId TRThreadWrapper::TypeId() const {
    return Item().type_id();
}

ui32 TRThreadWrapper::StartTimeInMinutes() const {
    return Item().tz_start_time() / 60;
}

TRThread::ETransportType TRThreadWrapper::TransportType() const {
    return Item().t_type_id();
}

// month : 1 - 12
// day: 1 - 31
bool TRThreadWrapper::RunsAt(ui32 month, ui32 day) const {
    Y_ASSERT(1 <= month && month <= 12 && 1 <= day && day <= 31); // В дебаге можно будет отловить, в проде не стоит
    return GetBit(Item().year_days(month - 1), BitsInMonth - day);
}

const TString& TRThreadWrapper::Number() const {
    return Item().number();
}

const TString& TRThreadWrapper::Title(ELanguage language, const TWrappedRaspDatabase& info) const {
    if (!Titles.contains(language)) {
        TString title = TDefaultTitleGenerator().GenerateTitle(*this, language, info);
        const_cast<THashMap<ELanguage, TString>&>(Titles).emplace(
            language, title);
    }
    return Titles.at(language);
}

const TString& TRThreadWrapper::Uid() const {
    return Uid_.GetRef();
}

object_id_t TRThreadWrapper::CompanyId() const {
    return Item().company_id();
}

object_id_t TRThreadWrapper::RouteId() const {
    return Item().route_id();
}

TThreadStationWrapper::TThreadStationWrapper(const TThreadStation& item, const TWrappedRaspDatabase& info)
    : TInnerIdWrapper<TThreadStation>(item, info)
{
    Timezone_ = info.GetTimezoneById(item.time_zone());
    if (info.Exists<TStation>(item.station_id()))
        Station_ = TMaybe<TStationWrapper>(info.GetItemWithOuterId<TStationWrapper>(item.station_id()));
    if (info.Exists<TRThread>(item.thread_id()))
        Thread_ = TMaybe<TRThreadWrapper>(info.GetItemWithOuterId<TRThreadWrapper>(item.thread_id()));
}

TThreadStationWrapper::TThreadStationWrapper(const TThreadStation& item,
                                             object_id_t pk,
                                             const NDatetime::TTimeZone& timezone,
                                             const TMaybe<TStationWrapper>& station,
                                             const TMaybe<TRThreadWrapper>& thread)
    : TInnerIdWrapper<TThreadStation>(item, pk)
    , Station_(station)
    , Thread_(thread)
    , Timezone_(timezone)
{
}

object_id_t TThreadStationWrapper::StationId() const {
    return Station_->Id();
}

object_id_t TThreadStationWrapper::ThreadId() const {
    return Thread_->Id();
}

bool TThreadStationWrapper::IsSearchableFrom() const {
    return Item().is_searchable_from();
}

bool TThreadStationWrapper::DepartureCodeSharing() const {
    return Item().departure_code_sharing();
}

bool TThreadStationWrapper::ArrivalCodeSharing() const {
    return Item().arrival_code_sharing();
}

TMaybe<i64> TThreadStationWrapper::DepartureOffset() const {
    if (Item().has_departure()) {
        return {Item().tz_departure()};
    }
    return {};
}

TMaybe<i64> TThreadStationWrapper::ArrivalOffset() const {
    if (Item().has_arrival()) {
        return {Item().tz_arrival()};
    }
    return {};
}

bool TThreadStationWrapper::IsTechnicalStop() const {
    return Item().is_technical_stop();
}

const NDatetime::TTimeZone& TThreadStationWrapper::Timezone() const {
    return Timezone_;
}

bool TThreadStationWrapper::IsSearchableTo() const {
    return Item().is_searchable_to();
}

TStation::EMajority TThreadStationWrapper::StationMajority() const {
    return Station_->MajorityId();
}

const TStationWrapper& TThreadStationWrapper::Station() const {
    return *Station_;
}

const TRThreadWrapper& TThreadStationWrapper::Thread() const {
    return *Thread_;
}

bool TThreadStationWrapper::HasSettlement() const {
    return Station().HasSettlement();
}

const TSettlementWrapper& TThreadStationWrapper::Settlement() const {
    return Station().Settlement();
}

bool TThreadStationWrapper::HasStation() const {
    return Station_.Defined();
}

bool TThreadStationWrapper::HasThread() const {
    return Thread_.Defined();
}

object_id_t TThreadStationWrapper::TerminalId() const {
    return Item().terminal_id();
}

TStationWrapper::TStationWrapper(const TStation& item, const TWrappedRaspDatabase& info)
    : TInnerIdWrapper<TStation>(item, info)
{
    Timezone_ = info.GetTimezoneById(item.time_zone());
    if (info.Exists<TSettlement>(item.settlement_id()))
        Settlement_ = TMaybe<TSettlementWrapper>(info.GetItemWithOuterId<TSettlementWrapper>(item.settlement_id()));
}

object_id_t TStationWrapper::SettlementId() const {
    return Settlement_->Id();
}

TStation::EMajority TStationWrapper::MajorityId() const {
    return Item().majority_id();
}

TStationWrapper::TStationWrapper(const TStation& item, object_id_t id,
                                 const TSettlement& settlement, object_id_t settlementId)
    : TInnerIdWrapper<TStation>(item, id)
{
    Settlement_.ConstructInPlace(settlement, settlementId);
}

const TSettlementWrapper& TStationWrapper::Settlement() const {
    return *Settlement_;
}

bool TStationWrapper::HasSettlement() const {
    return Settlement_.Defined();
}

bool TStationWrapper::IsAirport() const {
    return Item().is_airport();
}

const NDatetime::TTimeZone& TStationWrapper::Timezone() const {
    return Timezone_;
}

TStationToSettlementWrapper::TStationToSettlementWrapper(const TStation2Settlement& item, const TWrappedRaspDatabase& info)
    : TWrapper<TStation2Settlement>(item)
{
    SettlementId_ = info.GetInnerId<TSettlement>(item.settlement_id());
    StationId_ = info.GetInnerId<TStation>(item.station_id());
}

object_id_t TStationToSettlementWrapper::SettlementId() const {
    return SettlementId_;
}

object_id_t TStationToSettlementWrapper::StationId() const {
    return StationId_;
}

TString NRasp::GetFirstNotEmptyOrDefault(std::initializer_list<TString>&& words, const TString& defaultAnswer) {
    auto it = FindIf(words, [](auto& x) { return !x.empty(); });

    return it == words.end() ? defaultAnswer : *it;
}

TThreadTariffWrapper::TThreadTariffWrapper(const TThreadTariff& item, const TWrappedRaspDatabase& info)
    : TInnerIdWrapper(item, info)
{
    if (info.Exists<TStation>(item.station_from_id()))
        StationFrom_ = TMaybe<TStationWrapper>(info.GetItemWithOuterId<TStationWrapper>(item.station_from_id()));

    if (info.Exists<TStation>(item.station_to_id()))
        StationTo_ = TMaybe<TStationWrapper>(info.GetItemWithOuterId<TStationWrapper>(item.station_to_id()));

    if (info.Exists<TSettlement>(item.settlement_to_id()))
        SettlementTo_ = TMaybe<TSettlementWrapper>(info.GetItemWithOuterId<TSettlementWrapper>(item.settlement_to_id()));

    if (info.Exists<TSettlement>(item.settlement_from_id()))
        SettlementFrom_ = TMaybe<TSettlementWrapper>(info.GetItemWithOuterId<TSettlementWrapper>(item.settlement_from_id()));

    if (info.Exists<TSupplier>(item.supplier_id()))
        Supplier_ = TMaybe<TSupplierWrapper>(info.GetItemWithOuterId<TSupplierWrapper>(item.supplier_id()));

    if (info.Exists<TRThread>(item.thread_id()))
        Thread_ = TMaybe<TRThreadWrapper>(info.GetItemWithOuterId<TRThreadWrapper>(item.thread_id()));

    TimezoneFrom_ = info.GetTimezoneById(item.time_zone_from());
}

bool TThreadTariffWrapper::RunsAt(ui32 month, ui32 day) const {
    return GetBit(Item().year_days_from(month - 1), BitsInMonth - day);
}

const TStationWrapper& TThreadTariffWrapper::StationFrom() const {
    return *StationFrom_;
}

const TStationWrapper& TThreadTariffWrapper::StationTo() const {
    return *StationTo_;
}

const TSettlementWrapper& TThreadTariffWrapper::SettlementFrom() const {
    return *SettlementFrom_;
}

const TSettlementWrapper& TThreadTariffWrapper::SettlementTo() const {
    return *SettlementTo_;
}

const TSupplierWrapper& TThreadTariffWrapper::Supplier() const {
    return *Supplier_;
}

const TRThreadWrapper& TThreadTariffWrapper::Thread() const {
    return *Thread_;
}

const NDatetime::TTimeZone& TThreadTariffWrapper::TimezoneFrom() const {
    return TimezoneFrom_;
}

bool TThreadTariffWrapper::HasStationFrom() const {
    return StationFrom_.Defined();
}

bool TThreadTariffWrapper::HasStationTo() const {
    return StationTo_.Defined();
}

TMaybe<NDatetime::TSimpleTM> TThreadTariffWrapper::LocalDeparture(const NDatetime::TSimpleTM& date) const {
    using namespace NDatetime;

    auto departure = TSimpleTM(date.RealYear(), date.RealMonth(), date.MDay);
    departure.Add(TSimpleTM::F_MIN, DepartureInMinutes());

    TSimpleTM departureNaive, departureLocal;

    if (TimezoneFrom() == StationFrom().Timezone()) {
        departureNaive = departure;
        departureLocal = CreateCivilTime(StationFrom().Timezone(),
                                         departureNaive.RealYear(), departureNaive.RealMonth(), departureNaive.MDay);
    } else {
        bool founded = false;
        for (int offset : {0, -1, 1}) {
            departureNaive = departure;
            departureNaive.Add(TSimpleTM::F_DAY, offset);

            auto departureLocalUTC = CreateCivilTime(StationFrom().Timezone(),
                                                     departureNaive.RealYear(), departureNaive.RealMonth(), departureNaive.MDay);

            departureLocal = ToCivilTime(ToAbsoluteTime(departureLocalUTC, StationFrom().Timezone()), TimezoneFrom());
            if (departureLocal.RealYear() == date) {
                founded = true;
                break;
            }
        }

        if (!founded)
            return {};
    }

    if (!RunsAt(departureNaive.RealMonth(), departureNaive.MDay))
        return {};

    return departureLocal;
}

double TThreadTariffWrapper::Tariff() const {
    return Item().tariff();
}

TCurrencyProto::ECurrency TThreadTariffWrapper::Currency() const {
    return Item().currency();
}

TRThread::ETransportType TThreadTariffWrapper::TransportType() const {
    return Item().t_type();
}

TString TThreadTariffWrapper::TransportCode() const {
    switch (TransportType()) {
        case TRThread::TRAIN:
            return "train";
        case TRThread::PLANE:
            return "plane";
        case TRThread::BUS:
            return "bus";
        case TRThread::SUBURBAN:
            return "suburban";
        case TRThread::HELICOPTER:
            return "helicopter";
        case TRThread::WATER:
            return "water";
        default:
            ythrow yexception() << "Incorrect transport code for tariff [id=" << Item().id() << "]";
    }
}

i32 TThreadTariffWrapper::DepartureInMinutes() const {
    return Item().time_from() / 60;
}

bool TThreadTariffWrapper::HasThread() const {
    return Thread_.Defined();
}

TSupplierWrapper::TSupplierWrapper(const TSupplier& item, const TWrappedRaspDatabase& info)
    : TInnerIdWrapper(item, info)
{
}

bool TSupplierWrapper::CanByFrom(ELanguage language) const {
    if (language == LANG_RUS)
        return Item().can_buy_ru();
    if (language == LANG_UKR)
        return Item().can_buy_ua();
    if (language == LANG_TUR)
        return Item().can_buy_tr();
    return false;
}

const TString& TSupplierWrapper::Code() const {
    return Item().code();
}

const TString& TSupplierWrapper::Title() const {
    return Item().title();
}

const TString& TSupplierWrapper::SaleUrlTemplate() const {
    return Item().sale_url_template();
}

i32 TSupplierWrapper::SaleStartDays() const {
    return Item().sale_start_days();
}

i32 TSupplierWrapper::SaleStopHours() const {
    return Item().sale_stop_hours();
}
