#pragma once

#include "helpers.h"

#include <travel/rasp/route-search-api/proto/rthread.pb.h>
#include <travel/rasp/route-search-api/proto/rtstation.pb.h>
#include <travel/rasp/route-search-api/proto/settlement.pb.h>
#include <travel/rasp/route-search-api/proto/station.pb.h>
#include <travel/rasp/route-search-api/proto/station_to_settlement.pb.h>
#include <travel/rasp/route-search-api/proto/tariff.pb.h>
#include <travel/rasp/route-search-api/proto/supplier.pb.h>
#include <travel/rasp/route-search-api/proto/currency.pb.h>

#include <library/cpp/timezone_conversion/civil.h>
#include <library/cpp/langs/langs.h>

#include <util/generic/maybe.h>
#include <util/generic/hash_set.h>
#include <util/draft/datetime.h>

static NDatetime::TTimeZone MSK_TIMEZONE = NDatetime::GetTimeZone("Europe/Moscow");

namespace NRasp {
    // Обертки со сжатыми id и другими плюшками

    class TUnknownLanguageException: public yexception {
    };

    class TWrappedRaspDatabase;

    template <typename T>
    class TWrapper;
    template <typename T>
    class TInnerIdWrapper;
    class TSettlementWrapper;
    class TStationWrapper;
    class TRThreadWrapper;
    class TThreadStationWrapper;
    class TStationToSettlementWrapper;
    class TSupplierWrapper;

    template <class T>
    class TWrapper {
    public:
        using TBase = T;

        explicit TWrapper(const T& item)
            : Item_(&item)
        {
        }

        const T& Item() const {
            return *Item_;
        }

    protected:
        T const* Item_;
    };

    template <class T>
    class TInnerIdWrapper: public TWrapper<T> {
    public:
        TInnerIdWrapper(const T& item, const TWrappedRaspDatabase& info);

        TInnerIdWrapper(const T& item, object_id_t id);

        TInnerIdWrapper(const TInnerIdWrapper&) = default;

        TInnerIdWrapper& operator=(const TInnerIdWrapper&) = default;

        inline object_id_t Id() const noexcept {
            return InnerId;
        }

        inline object_id_t OuterId() const noexcept {
            return TWrapper<T>::Item().id();
        }

    protected:
        object_id_t InnerId;
    };

    template <class T>
    TInnerIdWrapper<T>::TInnerIdWrapper(const T& item, object_id_t id)
        : TWrapper<T>(item)
        , InnerId(id)
    {
    }

    class TSettlementWrapper: public TInnerIdWrapper<TSettlement> {
    public:
        TSettlementWrapper(const TSettlement& item, const TWrappedRaspDatabase& info);

        TSettlementWrapper(const TSettlement& item, object_id_t id,
                           const NDatetime::TTimeZone& timeZone = MSK_TIMEZONE);

        TString Title(ELanguage language) const;

        TString PopularTitle(ELanguage language) const;

        const NDatetime::TTimeZone& Timezone() const;

    private:
        NDatetime::TTimeZone Timezone_;
    };

    class TRThreadWrapper: public TInnerIdWrapper<TRThread> {
    public:
        TRThreadWrapper(const TRThread& item, const TWrappedRaspDatabase& info);
        TRThreadWrapper(const TRThread& item, object_id_t pk, const TString& uid = "");

        TRThread::ETypeId TypeId() const;

        TRThread::ETransportType TransportType() const;

        const TString& Number() const;

        const TString& Title(ELanguage language, const TWrappedRaspDatabase& info) const;

        ui32 StartTimeInMinutes() const;

        const TString& Uid() const;

        bool RunsAt(ui32 month, ui32 day) const;

        object_id_t CompanyId() const;

        object_id_t RouteId() const;

    private:
        THashMap<ELanguage, TString> Titles;
        TMaybe<TString> Uid_;
    };

    class TStationWrapper: public TInnerIdWrapper<TStation> {
    public:
        TStationWrapper(const TStation& item, const TWrappedRaspDatabase& info);

        TStationWrapper(const TStation& item, object_id_t id, const TSettlement&, object_id_t settlementId);

        TString PopularTitle(ELanguage language) const;

        TString Title(ELanguage language) const;

        bool HasSettlement() const;

        object_id_t SettlementId() const;

        const TSettlementWrapper& Settlement() const;

        TStation::EMajority MajorityId() const;

        bool IsAirport() const;

        const NDatetime::TTimeZone& Timezone() const;

    private:
        NDatetime::TTimeZone Timezone_;
        TMaybe<TSettlementWrapper> Settlement_;
    };

    class TThreadStationWrapper: public TInnerIdWrapper<TThreadStation> {
    public:
        TThreadStationWrapper(const TThreadStation& item, const TWrappedRaspDatabase& info);

        TThreadStationWrapper(const TThreadStation& item,
                              object_id_t pk,
                              const NDatetime::TTimeZone& timezone,
                              const TMaybe<TStationWrapper>& station,
                              const TMaybe<TRThreadWrapper>& thread);

        bool HasStation() const;

        object_id_t StationId() const;

        bool HasThread() const;

        object_id_t ThreadId() const;

        const TStationWrapper& Station() const;

        bool HasSettlement() const;

        const TSettlementWrapper& Settlement() const;

        const TRThreadWrapper& Thread() const;

        bool IsSearchableFrom() const;

        bool IsSearchableTo() const;

        bool DepartureCodeSharing() const;

        bool ArrivalCodeSharing() const;

        TMaybe<i64> DepartureOffset() const;

        TMaybe<i64> ArrivalOffset() const;

        bool IsTechnicalStop() const;

        const NDatetime::TTimeZone& Timezone() const;

        TStation::EMajority StationMajority() const;

        object_id_t TerminalId() const;

    private:
        TMaybe<TStationWrapper> Station_;
        TMaybe<TRThreadWrapper> Thread_;
        NDatetime::TTimeZone Timezone_;
    };

    class TStationToSettlementWrapper: public TWrapper<TStation2Settlement> {
    public:
        TStationToSettlementWrapper(const TStation2Settlement& item, const TWrappedRaspDatabase& info);

        object_id_t SettlementId() const;

        object_id_t StationId() const;

    private:
        object_id_t SettlementId_;
        object_id_t StationId_;
    };

    class TSupplierWrapper: public TInnerIdWrapper<TSupplier> {
    public:
        TSupplierWrapper(const TSupplier& item, const TWrappedRaspDatabase& info);

        const TString& Title() const;

        const TString& Code() const;

        bool CanByFrom(ELanguage language) const;

        const TString& SaleUrlTemplate() const;

        i32 SaleStartDays() const;

        i32 SaleStopHours() const;
    };

    class TThreadTariffWrapper: public TInnerIdWrapper<TThreadTariff> {
    public:
        TThreadTariffWrapper(const TThreadTariff& item, const TWrappedRaspDatabase& raspDatabase);

        bool RunsAt(ui32 month, ui32 day) const;

        const TStationWrapper& StationFrom() const;

        const TStationWrapper& StationTo() const;

        const TSettlementWrapper& SettlementFrom() const;

        bool HasStationFrom() const;

        const TSettlementWrapper& SettlementTo() const;

        bool HasStationTo() const;

        const TSupplierWrapper& Supplier() const;

        const TRThreadWrapper& Thread() const;

        bool HasThread() const;

        const NDatetime::TTimeZone& TimezoneFrom() const;

        double Tariff() const;

        TCurrencyProto::ECurrency Currency() const;

        TRThread::ETransportType TransportType() const;

        TString TransportCode() const;

        i32 DepartureInMinutes() const;

        TMaybe<NDatetime::TSimpleTM> LocalDeparture(const NDatetime::TSimpleTM& date) const;

    private:
        TMaybe<TStationWrapper> StationFrom_;
        TMaybe<TStationWrapper> StationTo_;
        TMaybe<TSettlementWrapper> SettlementFrom_;
        TMaybe<TSettlementWrapper> SettlementTo_;
        TMaybe<TSupplierWrapper> Supplier_;
        TMaybe<TRThreadWrapper> Thread_;
        NDatetime::TTimeZone TimezoneFrom_;
    };

    TString GetFirstNotEmptyOrDefault(std::initializer_list<TString>&& string, const TString& defaultAnswer = "");
}
