import logging
import os
import sys

import boto3
from botocore.client import Config

MDS_URL = "http://s3.mds.yandex.net"
MDS_ACCESS_KEY_ID = os.environ['MDS_ACCESS_KEY_ID']
MDS_ACCESS_SECRET_KEY = os.environ['MDS_ACCESS_SECRET_KEY']
MDS_BUCKET = os.environ.get('MDS_BUCKET', 'rasp-bucket')
MDS_CONNECT_TIMEOUT = 1
MDS_READ_TIMEOUT = 10
MDS_RETRIES_MAX_ATTEMPTS = 1
MDS_FOLDER = '/dumper'
MDS_PATH = 'rzd-proxy-dump'
FILE_PATH = '{}_dump.zip'.format(os.environ['DEPLOY_NODE_DC'])

log = logging.getLogger(__name__)


class MDSS3Wrapper(object):
    def __init__(self, bucket, url=MDS_URL, key=MDS_ACCESS_KEY_ID, secret_key=MDS_ACCESS_SECRET_KEY):
        self.bucket = bucket
        self.url = url
        self.key = key
        self.secret_key = secret_key
        self._client = None

    @property
    def client(self):
        if not self._client:
            self._client = get_mds_s3_raw_client(self.url, self.key, self.secret_key)

        return self._client

    def download(self, filepath, dirpath, mds_base_path):
        mds_path = os.path.join(mds_base_path, filepath)
        download_path = os.path.join(dirpath, filepath+'.dump')
        self.client.download_file(self.bucket, mds_path, download_path)

    def upload_file(self, filepath, dirpath, mds_base_path):
        mds_path = os.path.join(mds_base_path, filepath)
        file = os.path.join(dirpath, filepath)
        log.debug('Uploading file %s to bucket %s, path %s', file, self.bucket, mds_path)
        self.client.upload_file(Filename=file, Bucket=self.bucket, Key=mds_path)


def get_mds_s3_raw_client(url, key, secret_key):
    session = boto3.session.Session(aws_access_key_id=key, aws_secret_access_key=secret_key)
    config = Config(
        connect_timeout=MDS_CONNECT_TIMEOUT,
        read_timeout=MDS_READ_TIMEOUT,
        retries={'max_attempts': MDS_RETRIES_MAX_ATTEMPTS}
    )
    return session.client(service_name='s3', endpoint_url=url, config=config)


if __name__ == "__main__":
    client = MDSS3Wrapper(MDS_BUCKET)
    if sys.argv[1] == "upload":
        client.upload_file(FILE_PATH, MDS_FOLDER, MDS_PATH)
    elif sys.argv[1] == "download":
        client.download(FILE_PATH, MDS_FOLDER, MDS_PATH)
