# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from contextlib import contextmanager
from threading import local

import ibm_db
import settings


log = logging.getLogger(__name__)


connection_store = local()


class ManagerRequiredError(Exception):
    pass


class ManagerAlreadyInitializedError(Exception):
    pass


def get_raw_connection(db_name=None, host_name=None, port=None):
    """
    Про ошибки коннекта читать тут
    https://www.ibm.com/support/knowledgecenter/en/SSEPGG_11.1.0/com.ibm.db2.luw.messages.sql.doc/doc/rsqlmsg.html

    :param db_name:
    :param host_name:
    :param port:
    :return:
    """
    if db_name is None:
        db_name = settings.RZD_DATABASE_NAME

    if host_name is None:
        host_name = settings.RZD_HOSTNAME

    if port is None:
        port = settings.RZD_PORT

    connection_string = ';'.join([
        'DRIVER={IBM DB2 ODBC DRIVER}',
        'DATABASE={DATABASE_NAME}'.format(DATABASE_NAME=db_name),
        'HOSTNAME={HOSTNAME}'.format(HOSTNAME=host_name), 'PORT={}'.format(port),
        'PROTOCOL=TCPIP', 'UID=DGDPEX04', 'PWD=DGDP5396'
    ])
    log.info(u'Подключаемся к базе РЖД %s: %s', db_name, connection_string)
    connection = ibm_db.connect(connection_string, '', '')
    log.info(u'Успешно подключились к базе РЖД %s', db_name)
    log.info(u'Отключаем автокоммит')
    ibm_db.autocommit(connection, False)

    return connection


@contextmanager
def rzd_db_manager():
    try:
        if hasattr(connection_store, 'connection'):
            raise ManagerAlreadyInitializedError(u'Менеджер подключений для этой нитки уже инициализирован')
        connection_store.connection = None
        yield
    finally:
        if hasattr(connection_store, 'connection'):
            if ibm_db.active(connection_store.connection):
                ibm_db.close(connection_store.connection)
            del connection_store.connection


def get_connect(db_name=None, host_name=None, port=None):
    if not hasattr(connection_store, 'connection'):
        raise ManagerRequiredError(u'Нужно использовать rzd_db_manager для управляемых подключений')

    if not ibm_db.active(connection_store.connection):
        connection_store.connection = get_raw_connection(db_name, host_name, port)

    return connection_store.connection


def get_rowdicts_from_resource(resource):
    # Нужно сразу собрать все данные, чтобы коннект не отпал посреди вызова.
    # Такое почему-то случается.
    rowdicts = []

    while True:
        rowdict = ibm_db.fetch_assoc(resource)
        if not rowdict:
            break

        rowdicts.append(rowdict)

    return rowdicts


def repr_call_proc_param(param):
    if isinstance(param, basestring):
        return u"'{}'".format(param)
    else:
        return unicode(param)


def rzd_sql_callproc(connect, procedure, params):
    log.info(u'Запускаем РЖД SQL PROCEDURE: %s (%s)', procedure, u', '.join(map(repr_call_proc_param, params)))
    return ibm_db.callproc(
        connect, procedure,
        params
    )
