#include <travel/hotels/lib/cpp/ordinal_date/ordinal_date.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/protobuf_object_writer.h>
#include <travel/proto/dicts/rasp/transport.pb.h>
#include <travel/rasp/scripts/build_direction_minimal_price/lib/direction_reducer.h>
#include <travel/rasp/scripts/build_direction_minimal_price/lib/helpers.h>
#include <travel/rasp/scripts/build_direction_minimal_price/lib/route_mapper.h>
#include <travel/rasp/scripts/build_direction_minimal_price/lib/route_reducer.h>
#include <travel/rasp/scripts/build_direction_minimal_price/proto/min_price.pb.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/temp_table.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/global/global.h>

#include <util/draft/datetime.h>

using namespace NYT;
using namespace NRasp::NData;
using namespace NRasp::NDumper;
using namespace NTravel::NOrdinalDate;

REGISTER_REDUCER(TRouteReducer);
REGISTER_MAPPER(TRouteMapper);
REGISTER_REDUCER(TDirectionReducer);

int main(int argc, const char* argv[]) {
    Initialize(argc, argv);

    size_t top;
    i32 minDateDelta;
    TString rawMinPriceByDirectory;
    TString ytServerName;
    TString outPath;

    NLastGetopt::TOpts opts = NLastGetopt::TOpts::Default();
    opts.AddLongOption("depth", "See [depth] days in the past")
        .StoreResult<size_t>(&top)
        .DefaultValue("100");
    opts.AddLongOption("min-date-delta", "Save only prices that not older than current_time - min_date_delta")
        .StoreResult<i32>(&minDateDelta)
        .DefaultValue("3");
    opts.AddLongOption("yt-directory", "Directory with min prices by route")
        .StoreResult<TString>(&rawMinPriceByDirectory)
        .DefaultValue("//home/rasp/rasp/min_price/production/rasp-min-prices-by-routes");
    opts.AddLongOption("yt-server-name", "Yt servert name")
        .StoreResult<TString>(&ytServerName)
        .DefaultValue("hahn");
    opts.AddLongOption("out", "Path for saving min price by direction dump")
        .StoreResult<TString>(&outPath)
        .Required();

    NLastGetopt::TOptsParseResult optsResult(&opts, argc, argv);

    auto minDate = FromInstant(TInstant::Now() - TDuration::Days(minDateDelta));

    INFO_LOG << "Start" << Endl;
    IClientPtr client = CreateClient(ytServerName);
    auto routesTable = TTempTable(client, "prepared_routes");
    auto routesTableName = routesTable.Name();
    INFO_LOG << "Map min price data by route from raw to clean" << Endl;
    {
        auto spec = TMapOperationSpec();
        for (auto& table : GetLastTables(client, rawMinPriceByDirectory, top)) {
            spec.AddInput<TNode>(table);
        }
        spec.AddOutput<TNode>(routesTableName);

        client->Map(spec, new TRouteMapper(minDate));
    }
    INFO_LOG << "Prepare to route reduce" << Endl;
    {
        TVector<TString> fields = {
            "type",
            "date_forward",
            "from_settlement_id",
            "from_station_id",
            "to_settlement_id",
            "to_station_id",
            "route_uid",
            "timestamp",
        };
        auto spec = TSortOperationSpec();
        spec.AddInput(routesTableName);
        spec.Output(routesTableName);
        spec.SortBy(fields);

        client->Sort(spec);
    }

    auto minPriceByRouteTable = TTempTable(client, "min_price_by_route");
    INFO_LOG << "Make updated uniq routes" << Endl;
    {
        TVector<TString> fields = {
            "type",
            "date_forward",
            "from_settlement_id",
            "from_station_id",
            "to_settlement_id",
            "to_station_id",
            "route_uid",
        };
        auto spec = TReduceOperationSpec();
        spec.AddInput<TNode>(routesTableName);
        spec.AddOutput<TNode>(minPriceByRouteTable.Name());
        spec.ReduceBy(fields);
        client->Reduce(spec, new TRouteReducer());
    }
    INFO_LOG << "Prepare to direction reduce" << Endl;
    {
        TVector<TString> fields = {
            "type",
            "date_forward",
            "from_settlement_id",
            "from_station_id",
            "to_settlement_id",
            "to_station_id",
            "price",
        };
        auto spec = TSortOperationSpec();
        spec.AddInput(minPriceByRouteTable.Name());
        spec.Output(minPriceByRouteTable.Name());
        spec.SortBy(fields);

        client->Sort(spec);
    }
    INFO_LOG << "Make direction min prices" << Endl;
    auto minPriceByDirection = TTempTable(client, "min_price_by_route");
    {
        TVector<TString> fields = {
            "type",
            "date_forward",
            "from_settlement_id",
            "from_station_id",
            "to_settlement_id",
            "to_station_id",
        };
        auto spec = TReduceOperationSpec();
        spec.AddInput<TNode>(minPriceByRouteTable.Name());
        spec.AddOutput<TNode>(minPriceByDirection.Name());
        spec.ReduceBy(fields);
        client->Reduce(spec, new TDirectionReducer());
    }

    INFO_LOG << "Save min prices data" << Endl;
    TProtobufWriter<TMinPrice> writer{outPath};
    auto reader = client->CreateTableReader<TNode>(minPriceByDirection.Name());
    for (; reader->IsValid(); reader->Next()) {
        auto& row = reader->GetRow();

        TMinPrice minPrice;
        minPrice.SetTransportType(static_cast<TTransport_EType>(row.At("type").AsInt64()));
        minPrice.SetDate(row.At("date_forward").AsInt64());
        minPrice.SetFromSettlementId(row.At("from_settlement_id").AsInt64());
        minPrice.SetFromStationId(row.At("from_station_id").AsInt64());
        minPrice.SetToSettlementId(row.At("to_settlement_id").AsInt64());
        minPrice.SetToStationId(row.At("to_station_id").AsInt64());
        minPrice.SetPrice(row.At("price").AsDouble());

        writer.Write(minPrice);
        // TODO(mangin): reflect station_id to settlement_id for train and bus prices
    }

    INFO_LOG << "Done" << Endl;
}
