#include "route_mapper.h"
using namespace NYT;

TRouteMapper::TRouteMapper(NTravel::NOrdinalDate::TOrdinalDate minDate)
    : MinDate_(minDate)
{
}

void TRouteMapper::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    for (; input->IsValid(); input->Next()) {
        const TNode& row = input->GetRow();
        TNode result;

        auto transportType = ParseTransportType(row.At("type").AsString());

        result["type"] = static_cast<i32>(transportType);
        NTravel::NOrdinalDate::TOrdinalDate dateForward = 0;
        try {
            dateForward = NTravel::NOrdinalDate::FromString(row.At("date_forward").AsString());
        } catch (NTravel::NOrdinalDate::TInvalidDateException) {
        }
        result["date_forward"] = dateForward;

        FillIds(row, result, "from");
        FillIds(row, result, "to");

        result["route_uid"] = row.At("route_uid");

        time_t timestamp = 0;
        try {
            timestamp = TInstant::ParseIso8601(row.At("timestamp").AsString()).TimeT();
        } catch (TDateTimeParseException) {
        }

        result["timestamp"] = timestamp;
        result["price"] = FromString<double>(row.At("price").AsString());

        bool skip = false;
        if (transportType == NRasp::NData::TTransport::TYPE_UNKNOWN) {
            skip = true;
        }
        if (result["from_settlement_id"] == 0 && result["from_station_id"] == 0 && result["to_settlement_id"] == 0 && result["to_station_id"] == 0) {
            skip = true;
        }
        auto& klass = row.At("class").AsString();
        if (klass == "first" || klass == "business") {
            skip = true;
        }
        if (dateForward < MinDate_) {
            skip = true;
        }
        if (timestamp = 0) {
            skip = true;
        }
        result["skip"] = skip;

        output->AddRow(result);
    }
}

void TRouteMapper::FillIds(const TNode& origin, TNode& destination, const TString& direction) const {
    auto& pointType = origin.At("object_" + direction + "_type");
    auto pointId = FromString<i32>(origin["object_" + direction + "_id"].AsString());

    auto settlementId = 0;
    auto stationId = 0;
    if (pointType == "SettlementTuple") {
        settlementId = pointId;
    } else if (pointType == "StationTuple" || pointType == "Station") {
        stationId = pointId;
    }

    destination[direction + "_settlement_id"] = settlementId;
    destination[direction + "_station_id"] = stationId;
}

NRasp::NData::TTransport_EType TRouteMapper::ParseTransportType(const TString& code) const {
    if (code == "train") {
        return NRasp::NData::TTransport::TYPE_TRAIN;
    } else if (code == "bus") {
        return NRasp::NData::TTransport::TYPE_BUS;
    } else if (code == "plane") {
        return NRasp::NData::TTransport::TYPE_PLANE;
    } else {
        return NRasp::NData::TTransport::TYPE_UNKNOWN;
    }
}
