# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import re

from django.conf import settings
from requests import Session
from zeep import Client, Settings
from zeep.helpers import serialize_object  # noqa
from zeep.transports import Transport

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.exceptions import SimpleUnicodeException
from common.utils.yasmutil import Metric, MeasurableDecorator


log = logging.getLogger(__name__)


define_setting('AEROEXPRESS_PORT', {
    Configuration.PRODUCTION: '30084'
}, default='48000')

AEROEXPRESS_URL = 'https://api.aeroexpress.ru:{port}/aeapi/services/TicketServiceService?wsdl'.format(
    port=settings.AEROEXPRESS_PORT)


DOCUMENT_REGEXPS_BY_TYPE = {
    'RUSSIAN PASSPORT': [r'^(\d{3})(\d{3})(\d{4})$', r'^(\w{1})(\w{1})(\d{3})$'],
    'ANOTHER IDENTITY DOCUMENT': [r'^([\u0410-\u045F\w]{3})([\u0410-\u045F\w]{1,13})([\u0410-\u045F\w]{4})$']
}


class AeroExpressClientError(SimpleUnicodeException):
    pass


class AeroExpressClientRequestError(AeroExpressClientError):
    pass


EXCEPTION_NAME_TO_METRIC = {
    AeroExpressClientError: 'client_errors_cnt',
    AeroExpressClientRequestError: 'request_errors_cnt'
}


class measurable(MeasurableDecorator):
    prefix = 'aeroex'

    def _handle_error(self, exc):
        result = super(measurable, self)._handle_error(exc)
        exc_type = type(exc)
        if exc_type in EXCEPTION_NAME_TO_METRIC:
            metric_name = EXCEPTION_NAME_TO_METRIC[type(exc)]
            result.extend([
                Metric(self._name('errors_cnt'), 1, 'ammm'),
                Metric(self._name(metric_name), 1, 'ammm'),
                Metric('errors_cnt', 1, 'ammm'),
                Metric(metric_name, 1, 'ammm')
            ])
            return result
        return result


class AeroExpressClient(object):
    def __init__(self, url, crt_file, key_file):
        self.url = url
        try:
            session = Session()
            session.cert = (crt_file, key_file)
            transport = Transport(session=session)
            zeep_settings = Settings(strict=False)
            self.client = Client(
                self.url,
                transport=transport,
                settings=zeep_settings
            )
        except Exception as ex:
            log.error(repr(ex) + str(crt_file) + str(key_file))
            log.error(url)
            raise AeroExpressClientError('Не получилось создать клиент Аэроэкспресса. {}'.format(repr(ex)))

    def _call(self, method, error_message='', **kwargs):
        try:
            response = getattr(self.client.service, method)(**kwargs)
        except Exception as ex:
            log.error(repr(ex))
            raise AeroExpressClientRequestError('Произошла ошибка. {}{}'.format(error_message, repr(ex)))

        return response

    @staticmethod
    def mask(document_number, regexps):
        for regexp in regexps:
            matched = re.match(regexp, document_number)
            if matched:
                masked_number = '{}{}{}'.format(matched.group(1), '*' * len(matched.group(2)), matched.group(3))
                return masked_number

    @staticmethod
    def get_mask_doc_id(doc_number, doc_type):
        return AeroExpressClient.mask(doc_number, DOCUMENT_REGEXPS_BY_TYPE[doc_type])

    def get_version_info(self):
        version_info = self._call('getVersionInfo', error_message='Не получилось узнать версию АПИ.')
        return version_info

    def get_document_types(self):
        document_types = self._call(
            'listPersonalInfoDocumentTypes',
            error_message='Не получилось узнать список документов.'
        )
        return document_types['documentType']

    def get_agent_balance(self):
        agent_balance = self._call('getAgentBalance', error_message='Не получилось узнать баланс агента.')
        return agent_balance

    @measurable()
    def get_menu(self, menuId=None):
        menu = self._call('getWwwMenu', menuId=menuId, error_message='Не получилось узнать меню продаж.')
        return menu['item']

    def get_daily_report(self, reportDate):
        report = self._call('getDailyReport', reportDate=reportDate, error_message='Не получилось получить отчет.')
        return report['ticket']


@measurable()
def get_client(crt_file, key_file, url=None):
    if url is None:
        url = AEROEXPRESS_URL
    return AeroExpressClient(url=url, crt_file=crt_file, key_file=key_file)
