# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta

import six
from django.conf import settings
from mongoengine.fields import (
    IntField, StringField, EmbeddedDocumentField, BooleanField, DecimalField, ListField, DateTimeField
)

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date.environment import now

from travel.rasp.suburban_selling.selling.abstract_models import (
    AbstractDocument, AbstractEmbeddedDocument, AbstractDynamicEmbeddedDocument
)


TRIP_COUNT_BY_ID = {
    2: 2
}

STANDARD = {
    'description': 'Одна поездка в вагоне Стандартного класса по любому маршруту аэроэкспресса в течение 30 '
                   'суток с даты, которую вы указали при покупке билета. '
                   'Действует при покупке за три дня и менее до даты поездки.'
}

define_setting('TARIFFS_ADDITIONS', {Configuration.PRODUCTION: {9: STANDARD}}, default={14: STANDARD})


class AeroBilling(object):
    contract_service_id = 607
    partner_id = '37008957'
    product_id = 'aeroex_id'


@six.python_2_unicode_compatible
class AeroexMenu(AbstractDocument):
    menu_id = IntField()
    name = StringField()
    last_id = BooleanField()
    price = DecimalField()
    label = StringField()
    order_type = IntField()
    max_tickets = IntField()
    max_days = IntField()
    description = StringField()

    meta = {
        'indexes': [
            {
                'fields': ['menu_id'],
                'name': 'menu_id_index'
            }
        ]
    }

    def __str__(self):
        return '{}__{}'.format(self.menu_id, self.name)

    def __repr__(self):
        return self.__str__()

    def __hash__(self):
        return hash(self.__str__())

    def get_trip_count(self):
        return TRIP_COUNT_BY_ID.get(self.menu_id, 1)

    def get_description(self):
        add_description = settings.TARIFFS_ADDITIONS.get(self.menu_id, {}).get('description', None)
        return add_description if add_description else self.description


@six.python_2_unicode_compatible
class TariffKey(AbstractEmbeddedDocument):
    station_from = IntField()
    station_to = IntField()
    express_type = StringField()
    company = IntField()

    def __str__(self):
        return '{}__{}__{}__{}'.format(self.station_from, self.station_to, self.express_type, self.company)

    def __repr__(self):
        return self.__str__()

    def __hash__(self):
        return hash(self.__str__())


class TariffData(AbstractDynamicEmbeddedDocument):
    pass


@six.python_2_unicode_compatible
class SuburbanTariff(AbstractDocument):
    key = EmbeddedDocumentField(TariffKey)
    data = EmbeddedDocumentField(TariffData)

    def __str__(self):
        return self.key.__str__()

    def __repr__(self):
        return self.__str__()

    def __hash__(self):
        return hash(self.__str__())

    meta = {
        'indexes': [
            {
                'fields': ['key'],
                'name': 'key_index'
            }
        ]
    }


class ClientContract(AbstractEmbeddedDocument):
    start_dt = DateTimeField()
    finish_dt = DateTimeField()
    is_active = BooleanField(default=False)
    is_cancelled = BooleanField(default=False)
    is_suspended = BooleanField(default=False)
    services = ListField(IntField())
    partner_commission_sum = DecimalField(precision=2)


class ClientContracts(AbstractDocument):
    updated_at = DateTimeField()
    partner_id = StringField()
    contracts = ListField(EmbeddedDocumentField(ClientContract))

    meta = {'indexes': [{
        'fields': ['updated_at', 'partner_id', 'contracts.services', 'contracts.is_active'],
        'name': 'client_contracts_index'
    }]}

    @classmethod
    def has_active_contract(cls):
        dt_now = now()
        update_client_contracts_frequency = 6
        contracts = list(cls.objects(__raw__={
            'contracts': {'$elemMatch': {
                'is_active': True,
                'services': {'$in': [AeroBilling.contract_service_id]}
            }},
            'updated_at': {
                # Берем на час больше на всякий случай
                '$gte': dt_now - timedelta(hours=update_client_contracts_frequency + 1),
                '$lt': dt_now + timedelta(days=1)
            },
            'partner_id': AeroBilling.partner_id}))

        return bool(contracts)
