# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from typing import Dict, Optional, List, Tuple, Iterable
from collections import defaultdict
from datetime import datetime, timedelta, time

from django.conf import settings
from mongoengine import Q

from common.dynamic_settings.default import conf
from common.models.tariffs import TariffTypeCode, SuburbanSellingFlow, SuburbanTariffProvider
from common.settings.utils import define_setting
from common.utils.date import MSK_TZ
from travel.library.python.tracing.instrumentation import traced_function

from travel.rasp.suburban_selling.selling.aeroexpress.models import ClientContracts
from travel.rasp.suburban_selling.selling.tariffs.tariffs_configuration import TariffsConfiguration
from travel.rasp.suburban_selling.selling.tariffs.selling_companies import SuburbanCarrierCode
from travel.rasp.suburban_selling.selling.aeroexpress.models import SuburbanTariff as AeroexTariff
from travel.rasp.suburban_selling.selling.tariffs.interfaces import (
    TariffsProvider, TariffKey, SellingTariff, TariffKeyData, TariffKeyDataStatus
)

log = logging.getLogger(__name__)


define_setting('AEROEX_TICKET_VALID_DAYS', default=30)
define_setting('AEROEX_TICKET_VALID_HOURS_IN_LAST_DAY', default=3)


class AeroexSellingTariff(SellingTariff):
    def __init__(self, tariff_type, name, description, price, max_days, valid_from, valid_until, book_data):
        # type: (str, str, str, float, Optional[int], datetime, datetime, Dict) -> None
        super(AeroexSellingTariff, self).__init__(
            SuburbanCarrierCode.AEROEXPRESS, tariff_type, name, description, price,
            max_days, valid_from, valid_until, book_data, SuburbanSellingFlow.AEROEXPRESS
        )

    def to_json(self):
        result = super(AeroexSellingTariff, self).to_json()
        # Поля переехали в book_data и здесь остаются лишь для обратной совместимости
        result['menu_id'] = self.book_data['menu_id']
        result['order_type'] = self.book_data['order_type']
        return result


class AeroexTariffsProvider(TariffsProvider):
    def __init__(self, tariffs_configuration):
        # type: (TariffsConfiguration) -> None
        super(AeroexTariffsProvider, self).__init__(tariffs_configuration)

    def get_selling_flows(self):
        return {SuburbanSellingFlow.AEROEXPRESS}

    @traced_function
    def get_tariffs(self, tariff_keys):
        # type: (Iterable[TariffKey]) -> Dict[TariffKey, TariffKeyData]

        if not conf.SUBURBAN_SELLING__AEROEX_TARIFFS_ENABLED:
            return {}
        if not ClientContracts.has_active_contract():
            return {}

        tariff_keys_filtered = {
            tk for tk in tariff_keys
            if SuburbanTariffProvider.AEROEXPRESS in
               self.tariffs_configuration.provider_codes_by_companies_id.get(tk.company, [])
        }
        aeroex_tariffs = self.get_aeroex_tariffs(tariff_keys_filtered)

        data_by_keys = {}  # type: Dict[TariffKey, TariffKeyData]

        selling_tariffs_by_from_to_date = {}
        for tariff_key in tariff_keys_filtered:
            data_by_keys[tariff_key] = TariffKeyData([], TariffKeyDataStatus.ACTUAL)
            from_to_key = (tariff_key.station_from, tariff_key.station_to)

            if from_to_key in aeroex_tariffs:
                for ae_tariff in aeroex_tariffs[from_to_key]:
                    if not ae_tariff:
                        continue

                    ae_tariff_data = ae_tariff.data
                    if not self._can_sell_on_date(ae_tariff_data, tariff_key.date):
                        continue

                    from_to_date_key = (tariff_key.station_from, tariff_key.station_to, tariff_key.date)

                    if from_to_date_key in selling_tariffs_by_from_to_date:
                        selling_tariff = selling_tariffs_by_from_to_date[from_to_date_key]
                    else:
                        valid_from, valid_until = self._get_valid_time(tariff_key.date)
                        selling_tariff = AeroexSellingTariff(
                            price=round(ae_tariff_data.price.to_decimal(), 2),
                            tariff_type=TariffTypeCode.AEROEXPRESS,
                            name=ae_tariff_data.name,
                            description=ae_tariff_data.description,
                            max_days=ae_tariff_data.max_days,
                            valid_from=valid_from,
                            valid_until=valid_until,
                            book_data=dict(
                                menu_id=ae_tariff_data.menu_id,
                                order_type=ae_tariff_data.order_type,
                            )
                        )
                        selling_tariffs_by_from_to_date[from_to_date_key] = selling_tariff

                    data_by_keys[tariff_key].tariffs.append(selling_tariff)

        return data_by_keys

    def get_aeroex_tariffs(self, tariff_keys):
        # type: (Iterable[TariffKey]) -> Dict[Tuple, List[AeroexTariff]]
        tariffs_query = Q()
        for tariff_key in tariff_keys:
            tariffs_query |= Q(key__station_from=tariff_key.station_from, key__station_to=tariff_key.station_to)

        aeroex_tariffs = defaultdict(list)
        for ae_tariff in AeroexTariff.objects.filter(tariffs_query):
            aeroex_tariffs[(ae_tariff.key.station_from, ae_tariff.key.station_to)].append(ae_tariff)

        return aeroex_tariffs

    def _can_sell_on_date(self, ae_tariff_data, tariff_date):
        dt = datetime.combine(tariff_date, datetime.min.time())
        return ae_tariff_data.begin_dt <= dt < ae_tariff_data.end_dt

    def _get_valid_time(self, tariff_date):
        valid_from = MSK_TZ.localize(datetime.combine(tariff_date, time(0)))
        valid_until = MSK_TZ.localize(
            datetime.combine(
                tariff_date + timedelta(days=settings.AEROEX_TICKET_VALID_DAYS),
                time(settings.AEROEX_TICKET_VALID_HOURS_IN_LAST_DAY)
            )
        )
        return valid_from, valid_until
