# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import namedtuple
from datetime import datetime, date
from typing import Dict, Iterable

import celery

from common.data_api.im.instance import im_client
from travel.rasp.library.python.api_clients.im import ImClient

from travel.rasp.suburban_selling.selling.im.models import BookData, ImTariffs

log = logging.getLogger(__name__)


ImTrainData = namedtuple(
    'ImTrainData',
    [
        'carrier_im_code', 'train_number', 'departure_dt', 'price',
        'im_provider', 'is_sale_forbidden', 'availability_indication'
    ]
)


def make_im_train_data_from_segment(segment):
    # type: (Dict[str]) -> ImTrainData
    """Обработка одного сегмента из выдачи ИМ, получение из него списка поездов"""

    car_group = segment['CarGroups'][0]

    return ImTrainData(
        carrier_im_code=segment['Carriers'][0],
        train_number=segment['TrainNumber'],
        departure_dt=segment['LocalDepartureDateTime'],
        price=car_group['MinPrice'],
        im_provider=segment['Provider'],
        is_sale_forbidden=segment['IsSaleForbidden'],
        availability_indication=car_group['AvailabilityIndication']
    )


def get_tariffs_data_from_provider(im_client, book_data):
    # type: (ImClient, BookData) -> Iterable[ImTrainData]

    info_str = 'tariffs for date={}, station_from_express_id={}, station_to_express_id={}'.format(
        book_data.date, book_data.station_from_express_id, book_data.station_to_express_id
    )
    log.info('Get {}'.format(info_str))

    try:
        trains = im_client.train_pricing(
            book_data.station_from_express_id,
            book_data.station_to_express_id,
            datetime.strptime(book_data.date, '%Y-%m-%d')
        ).get('Trains', [])

        trains_by_numbers = {}
        for segment in trains:
            if segment['IsSuburban']:
                im_train_data = make_im_train_data_from_segment(segment)
                trains_by_numbers[im_train_data.train_number] = im_train_data
        log.info('Received {}, number of suburban trains: {}'.format(info_str, len(trains)))

        return trains_by_numbers.values()

    except Exception:
        err_mess = 'Can\'t get {}'.format(info_str)
        log.exception(err_mess)
        return []


def save_data_to_cache(departure_date, station_from, station_to, trains):
    # type: (date, int, int, Iterable[ImTrainData]) -> None
    ImTariffs.objects(
        date=departure_date,
        station_from=station_from,
        station_to=station_to,
    ).update_one(
        im_trains=[
            {
                'carrier_im_code': train.carrier_im_code,
                'train_number': train.train_number,
                'departure_dt': train.departure_dt,
                'price': train.price,
                'im_provider': train.im_provider,
                'is_sale_forbidden': train.is_sale_forbidden,
                'availability_indication': train.availability_indication
            } for train in trains
        ],
        updated=datetime.utcnow(),
        upsert=True
    )

    log.info('Im tariffs saved to cache, from: {}, to: {}, date: {}'.format(date, station_from, station_to))


@celery.shared_task
def get_tariffs_from_im(departure_date, station_from, station_to):
    # type: (date, int, int) -> None
    """
    Получение списка поездов из выдачи ручки train_pricing провайдера,
    и запись полученных данных в кэш
    """
    im_tariffs = ImTariffs.objects.get(
        date=departure_date,
        station_from=station_from,
        station_to=station_to
    )
    trains = get_tariffs_data_from_provider(im_client, im_tariffs.book_data)
    save_data_to_cache(departure_date, station_from, station_to, trains)
