# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import pytz
from datetime import timedelta
from dateutil import parser
from typing import Dict

from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.api_clients.im import ImClient
from travel.rasp.library.python.api_clients.travel_api.providers.im import ImServiceBookData

from travel.rasp.suburban_selling.selling.order.helpers import (
    ForbiddenOrderError, SuburbanServiceFactory, WrongOrderDataError
)


class ImServiceFactory(SuburbanServiceFactory):
    def raise_if_order_forbidden(self):
        # type: () -> ()
        if not conf.SUBURBAN_SELLING__IM_ORDER_ENABLED:
            raise ForbiddenOrderError(
                'IM order is forbidden by the setting SUBURBAN_SELLING__IM_ORDER_ENABLED'
            )
        if not conf.SUBURBAN_SELLING_ENABLED:
            raise ForbiddenOrderError(
                'IM order is forbidden by the setting SUBURBAN_SELLING_ENABLED'
            )

    def make_book_data(self, order_data):
        # type: (Dict) -> ImServiceBookData

        book_data = order_data['book_data']

        today = ImClient.dt_to_im_date(environment.today() - timedelta(hours=1))
        if book_data['date'] < today:
            raise WrongOrderDataError('Order date is in the past: {}'.format(book_data['date']))
        if 'departure_dt' in book_data and 'departure_tz' in book_data:
            departure_pytz = pytz.timezone(book_data['departure_tz'])
            if departure_pytz.localize(parser.parse(book_data['departure_dt'])) < environment.now_aware():
                raise WrongOrderDataError('Last train of this day has gone {}'.format(book_data['departure_dt']))

        return ImServiceBookData(
            book_data['date'], int(book_data['station_from_express_id']), int(book_data['station_to_express_id']),
            book_data['train_number'], book_data.get('im_provider')
        )
