# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import pytz
from dateutil import parser
from datetime import datetime, timedelta, time
from typing import Dict, Iterable, List, Optional

from django.conf import settings
from mongoengine import Q

from common.dynamic_settings.default import conf
from common.settings.utils import define_setting
from common.models.tariffs import SuburbanSellingFlow
from common.models.geo import Station
from common.utils.date import MSK_TZ
from travel.rasp.library.python.common23.date import environment
from travel.library.python.tracing.instrumentation import traced_function

from travel.rasp.library.python.api_clients.im import ImClient

from travel.rasp.suburban_selling.selling.im.models import ImTariffs, ImTrain, IM_TRAIN_IS_AVAILABLE
from travel.rasp.suburban_selling.selling.im.get_from_im import get_tariffs_from_im
from travel.rasp.suburban_selling.selling.tariffs.from_to_key_cache_provider import (
    TariffFromToKey, FromToKeyCacheAsynchTariffsProvider, get_stations_express_codes, BaseFromToKeyTariffsData
)
from travel.rasp.suburban_selling.selling.tariffs.interfaces import SellingTariff, TariffKeyData
from travel.rasp.suburban_selling.selling.tariffs.tariffs_configuration import TariffsConfiguration


log = logging.getLogger(__name__)

define_setting('IM_TARIFFS_TTL', default=3600 * 24)
define_setting('IM_EMPTY_TARIFFS_TTL', default=900)

define_setting('IM_TICKET_VALID_DAYS', default=1)
define_setting('IM_TICKET_VALID_HOURS_IN_LAST_DAY', default=1)

IM_PROVIDER_CODE = 'im'


# Структуры, в которые загружаются данные о тарифах из ИМ или кэша в монге,
# и из которых потом значения сохраняются в кэш в монге
class ImTariff(object):
    def __init__(self, carrier, type_code, price, im_provider, station_from_tz):
        # type: (str, str, float, str, str) -> None
        self.carrier = carrier
        self.type_code = type_code
        self.price = price
        self.im_provider = im_provider
        self.station_from_tz = station_from_tz
        self.times_by_train_numbers = {}

    def get_canonical_train(self):
        # В качестве номера для book_data выбираем номер последнего по времени отправления поезда
        numbers = [(self.times_by_train_numbers[number], number) for number in self.times_by_train_numbers]
        latest_number = numbers[0]
        for n in numbers:
            if n[0] > latest_number[0]:
                latest_number = n
        departure_dt = parser.parse(latest_number[0])
        number = latest_number[1]
        return number, departure_dt


class ImBookData(object):
    def __init__(self, date, station_from_express_id, station_to_express_id):
        # type: (str, str, str) -> None
        self.date = date
        self.station_from_express_id = station_from_express_id
        self.station_to_express_id = station_to_express_id


class ImTariffsData(BaseFromToKeyTariffsData):
    def __init__(self, tariffs, book_data, updated=None, update_started=None):
        # type: (Iterable[ImTariff], ImBookData, Optional[datetime], Optional[datetime]) -> None
        super(ImTariffsData, self).__init__(
            tariffs_ttl=settings.IM_TARIFFS_TTL,
            empty_tariffs_ttl=settings.IM_EMPTY_TARIFFS_TTL,
            updated=updated,
            update_started=update_started
        )

        self.tariffs = tariffs
        self.book_data = book_data


class ImTariffsProvider(FromToKeyCacheAsynchTariffsProvider):
    """Провайдер для получения тарифов электричек из ИМ"""

    def __init__(self, tariff_type_ids_by_numbers, tariffs_configuration):
        # type: (Dict[str, int], TariffsConfiguration) -> None
        super(FromToKeyCacheAsynchTariffsProvider, self).__init__(tariffs_configuration)

        self.tariff_type_ids_by_numbers = tariff_type_ids_by_numbers

        self.carriers_by_im_code = {
            carrier.code_in_provider: carrier
            for carrier in self.tariffs_configuration.carriers_by_codes.values()
        }

    def get_selling_flows(self):
        return {SuburbanSellingFlow.SIMPLE}

    def check_get_tariffs_enabled(self):
        # type: () -> bool
        return conf.SUBURBAN_SELLING__IM_TARIFFS_ENABLED

    def get_train_numbers(self, complex_number):
        # type: (str) -> List[str]
        # Номер может прийти в формате вида 6543a/6544b
        numbers_str = ''.join(
            char for char in complex_number if char.isdigit() or char == '/'
        )
        numbers = numbers_str.split('/')
        return numbers

    def _define_tariff_type(self, tariff_type_by_ids, digit_numbers):
        tariff_type_code = None
        for digit_number in digit_numbers:
            cur_tariff_type = tariff_type_by_ids.get(self.tariff_type_ids_by_numbers.get(digit_number))
            if not cur_tariff_type:
                continue
            if tariff_type_code and cur_tariff_type.code != tariff_type_code:
                return None
            tariff_type_code = cur_tariff_type.code
        return tariff_type_code

    def make_tariffs_from_trains(self, im_trains, departure_tz):
        # type: (Iterable[ImTrain], str) -> List[ImTariff]

        dt_now = environment.now_aware()
        tariffs_dict = {}
        for train in im_trains:
            if train.is_sale_forbidden or train.availability_indication != IM_TRAIN_IS_AVAILABLE:
                continue

            if train.carrier_im_code not in self.carriers_by_im_code:
                continue

            try:
                if pytz.timezone(departure_tz).localize(parser.parse(train.departure_dt)) < dt_now:
                    continue
            except Exception:
                continue

            carrier_code = self.carriers_by_im_code[train.carrier_im_code].code
            tariff_type_by_ids = self.tariffs_configuration.tariff_types_by_carriers_and_ids[carrier_code]
            digit_numbers = self.get_train_numbers(train.train_number)

            tariff_type_code = self._define_tariff_type(tariff_type_by_ids, digit_numbers)
            if not tariff_type_code:
                continue

            tariff_key = (carrier_code, tariff_type_code)
            if tariff_key not in tariffs_dict:
                tariffs_dict[tariff_key] = ImTariff(
                    carrier_code, tariff_type_code, train.price, train.im_provider, departure_tz
                )
            im_tariff = tariffs_dict[tariff_key]

            im_tariff.times_by_train_numbers[train.train_number] = train.departure_dt
            if im_tariff.price != train.price:
                log.exception('Different prices ({} and {}) are found for tariff {}, provider {}'.format(
                    im_tariff.price, train.price, tariff_type_code, carrier_code
                ))
            if im_tariff.im_provider != train.im_provider:
                log.exception('Different IM providers ({} and {}) are found for tariff {}, provider {}'.format(
                    im_tariff.im_provider, train.im_provider, tariff_type_code, carrier_code
                ))

        return list(tariffs_dict.values())

    @traced_function
    def get_tariffs_data_from_cache(self, from_to_keys):
        # type: (Iterable[TariffFromToKey]) -> Dict[TariffFromToKey, ImTariffsData]
        """Получение тарифов из монги"""

        if not from_to_keys:
            return {}

        tariffs_query = Q()
        for key in from_to_keys:
            tariffs_query |= Q(
                date=key.date,
                station_from=key.station_from,
                station_to=key.station_to
            )

        keys_with_tariffs = {}
        for tariffs_doc in ImTariffs.objects.filter(tariffs_query):
            from_to_key = TariffFromToKey(
                date=tariffs_doc.date,
                station_from=tariffs_doc.station_from,
                station_to=tariffs_doc.station_to,
            )

            book_data = ImBookData(
                date=tariffs_doc.book_data.date,
                station_from_express_id=tariffs_doc.book_data.station_from_express_id,
                station_to_express_id=tariffs_doc.book_data.station_to_express_id,
            )

            departure_tz = Station.objects.get(id=from_to_key.station_from).time_zone
            tariffs = self.make_tariffs_from_trains(tariffs_doc.im_trains, departure_tz)

            keys_with_tariffs[from_to_key] = ImTariffsData(
                tariffs=tariffs,
                book_data=book_data,
                updated=tariffs_doc.updated,
                update_started=tariffs_doc.update_started
            )

        return keys_with_tariffs

    def prepare_base_tariff_key_data_in_cache(self, from_to_key):
        # type: (TariffFromToKey) -> Optional[str]
        """
        Сохранение в монгу данных по ключу тарифов перед запросом к провайдеру
        : returns: id обновленного документа из монги
        """
        express_ids = get_stations_express_codes(from_to_key)
        if not express_ids:
            return None
        station_from_express_id, station_to_express_id = express_ids
        book_data = {
            'date': ImClient.dt_to_im_date(from_to_key.date),
            'station_from_express_id': station_from_express_id,
            'station_to_express_id': station_to_express_id,
        }

        ImTariffs.objects(
            date=from_to_key.date,
            station_from=from_to_key.station_from,
            station_to=from_to_key.station_to,
        ).update_one(
            book_data=book_data,
            update_started=datetime.utcnow(),
            upsert=True
        )
        im_tariffs = ImTariffs.objects.get(
            date=from_to_key.date,
            station_from=from_to_key.station_from,
            station_to=from_to_key.station_to
        )
        return im_tariffs['id']

    def run_get_tariffs_data_from_provider(self, from_to_keys):
        # type: (List[TariffFromToKey]) -> None
        """Запуск асинхронного получения данных от провайдера, для тех ключей, для которых нет актуальных данных"""

        if not conf.SUBURBAN_SELLING__IM_API_CALL_ENABLED:
            return

        for from_to_key in from_to_keys:
            tariffs_data_id = self.prepare_base_tariff_key_data_in_cache(from_to_key)

            log.info('Async get tariffs from IM. IM tariffs data id: {}, from: {}, to: {}, date: {}'.format(
                tariffs_data_id, from_to_key.station_from, from_to_key.station_to, from_to_key.date
            ))

            get_tariffs_from_im.delay(from_to_key.date, from_to_key.station_from, from_to_key.station_to)

    @traced_function
    def make_selling_tariffs_from_tariffs_data(self, tariffs_data_for_all_keys):
        # type: (Dict[TariffFromToKey, ImTariffsData]) -> Dict[TariffFromToKey, TariffKeyData]
        """Формирование выходных данных по полученным тарифам"""

        data_by_from_to_keys = {}  # type: Dict[TariffFromToKey, TariffKeyData]
        for from_to_key, tariffs_data in tariffs_data_for_all_keys.items():
            data_by_from_to_keys[from_to_key] = TariffKeyData([], tariffs_data.status)

            tariffs_dict = self.tariffs_configuration.tariff_types_by_carriers_and_codes
            for tariff in tariffs_data.tariffs:
                tariff_date = ImClient.date_from_im_date(tariffs_data.book_data.date)
                valid_from = MSK_TZ.localize(datetime.combine(tariff_date, time(0)))
                valid_until = MSK_TZ.localize(
                    datetime.combine(
                        tariff_date + timedelta(days=settings.IM_TICKET_VALID_DAYS),
                        time(settings.IM_TICKET_VALID_HOURS_IN_LAST_DAY)
                    )
                )
                train_number, departure_dt = tariff.get_canonical_train()

                selling_tariff = SellingTariff(
                    partner=tariff.carrier,
                    tariff_type=tariff.type_code,
                    name=tariff.type_code,
                    description=tariffs_dict[tariff.carrier][tariff.type_code].description,
                    price=round(tariff.price, 2),
                    max_days=None,
                    valid_from=valid_from,
                    valid_until=valid_until,
                    book_data=dict(
                        date=tariffs_data.book_data.date,
                        station_from_express_id=tariffs_data.book_data.station_from_express_id,
                        station_to_express_id=tariffs_data.book_data.station_to_express_id,
                        train_number=train_number,
                        departure_dt=departure_dt.isoformat(),
                        departure_tz=tariff.station_from_tz,
                        im_provider=tariff.im_provider
                    ),
                    selling_flow=SuburbanSellingFlow.SIMPLE
                )
                data_by_from_to_keys[from_to_key].tariffs.append(selling_tariff)

        return data_by_from_to_keys
