# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import hashlib

import six
from dateutil import parser
from django.conf import settings
from django.utils.encoding import force_bytes
from mongoengine import (
    IntField, FloatField, StringField, DateTimeField, DynamicField, EmbeddedDocumentField, ListField, DecimalField,
    BooleanField
)
from pymongo.collation import Collation, CollationStrength

from travel.rasp.library.python.common23.date.environment import now_utc

from travel.rasp.suburban_selling.selling.aeroexpress.client import AeroExpressClient
from travel.rasp.suburban_selling.selling.abstract_models import AbstractDocument, AbstractEmbeddedDocument


class OrderStatus(object):
    INIT = 'init'
    RESERVED = 'reserved'
    CANCELLED = 'cancelled'
    START_PAYMENT_FAILED = 'start_payment_failed'
    START_PAYMENT = 'start_payment'
    PAYMENT_FAILED = 'payment_failed'
    PAYMENT_OUTDATED = 'payment_outdated'
    CONFIRM_FAILED = 'confirm_failed'
    TICKETS_UPDATED = 'tickets_updated'
    UPDATE_FAILED = 'update_failed'
    CONFIRMED = 'confirmed'
    PAID = 'paid'
    DONE = 'done'
    FAILED = 'failed'


class ApiOrderStatus(object):
    INIT = 'init'
    RESERVED = 'reserved'
    CONFIRMED = 'confirmed'
    DONE = 'done'
    FAILED = 'failed'
    CANCELLED = 'cancelled'


class Ticket(AbstractEmbeddedDocument):
    code = StringField(required=True)
    ticket_id = IntField(required=True)
    ticket_guid = StringField(required=True)
    ticket_url = StringField(required=True)
    token = StringField(required=True)
    trip_date = StringField(required=True)
    trip_time = StringField(required=True)
    st_arrival = StringField(required=True)
    st_depart = StringField(required=True)
    tariff = StringField(required=True)
    ticket_price = DecimalField(required=True)
    valid_until = StringField(required=True)


class BillingPayment(AbstractEmbeddedDocument):
    clear_at = DateTimeField()
    purchase_token = StringField(verbose_name='Идентификатор корзины в TRUST')
    payment_url = StringField(verbose_name='URL для iframe-а оплаты')
    create_payment_counter = IntField(default=0)
    last_payment_created_at = DateTimeField()
    status = StringField()
    resp_code = StringField()
    resp_desc = StringField()
    order_ids_list = ListField(StringField())


class Passenger(AbstractEmbeddedDocument):
    first_name = StringField()
    surname = StringField()
    patronymic_name = StringField()
    doc_type = StringField()
    doc_number_hash = StringField()
    doc_number_mask_hash = StringField()
    ticket = EmbeddedDocumentField(Ticket)

    @property
    def fio(self):
        return '{} {} {}'.format(self.surname, self.firstName, self.patronymicName)


class UserInfo(AbstractEmbeddedDocument):
    email = StringField(required=True)
    phone = StringField(required=True)
    passport_uid = IntField(default=None)
    region_id = IntField(default=None)
    ip = StringField(default=None)
    lat = StringField(default=None)
    lng = StringField(default=None)


class AeroexpressRequestData(AbstractEmbeddedDocument):
    menu_id = IntField(required=True)
    order_type = IntField(required=True)
    depart_date = StringField(required=True)


class ReservingProcess(AbstractEmbeddedDocument):
    reserve_try_counter = IntField(default=0)


def collation_index(field, name):
    return {
        'fields': [field],
        'name': name,
        'collation': Collation(locale=settings.LANGUAGE_CODE, strength=CollationStrength.PRIMARY)
    }


def hash_string(value, salt):
    return hashlib.sha256(b''.join(map(force_bytes, (settings.SECRET_KEY, salt, value)))).hexdigest()


def hash_doc_id(doc_id):
    return hash_string(doc_id, 'passenger_doc_id')


@six.python_2_unicode_compatible
class Order(AbstractDocument):
    meta = {
        'indexes': [
            'uid',
            'status',
            'api_status',
            'created_at',
            'aeroexpress_request_data.depart_date',
            'user_info.phone',
            'user_info.passport_uid',
            'passengers.doc_number_hash',
            collation_index('passengers.first_name', name='first_name_collation'),
            collation_index('passengers.surname', name='surname_collation'),
            collation_index('passengers.patronymic_name', name='patronymic_name_collation'),
            collation_index('user_info.email', name='user_email_collation'),
            'process.external_events_count',
            'process.lock_modified',
            'process.lock_uid',
            'process.state',
        ]
    }

    process = DynamicField(default={})  # workflow data
    created_at = DateTimeField(default=now_utc)
    status = StringField(required=True)
    api_status = StringField(required=True)

    finished_at = DateTimeField(help_text='Время завершения')

    uid = StringField(max_length=64, min_length=32, unique=True, required=True)

    reserve = EmbeddedDocumentField(ReservingProcess, required=True, default=ReservingProcess)

    reserved_until = DateTimeField(default=None)
    order_id = IntField()  # order id from partner (eg. Aeroex)
    summ = FloatField()

    station_from_id = IntField(required=True)
    station_to_id = IntField(required=True)
    passengers = ListField(EmbeddedDocumentField(Passenger))
    user_info = EmbeddedDocumentField(UserInfo, required=True)
    aeroexpress_request_data = EmbeddedDocumentField(AeroexpressRequestData, required=True)
    billing_payments = ListField(EmbeddedDocumentField(BillingPayment), default=[BillingPayment()],
                                 verbose_name='Корзины в TRUST. Нулевая == текущая.')
    updated_from_billing_at = DateTimeField(help_text='Время обновления статусов и расшифровок платежей из биллинга')
    updated_from_aeroexpress = DateTimeField(help_text='Время обновления информации из аэроэкспресса')
    request_departure_date = DateTimeField(help_text='Дата поездки')

    user = StringField(null=False)
    price = FloatField(null=False)

    error_is_sent = BooleanField(default=False)

    def __str__(self):
        return '{}__{}'.format(self.uid, self.price)

    def __repr__(self):
        return self.__str__()

    @property
    def current_billing_payment(self):
        return self.billing_payments[0]

    @classmethod
    def create_order(cls, order_data, reserve_data):
        if 'personal_info_data' in order_data:
            passengers = [Passenger(
                first_name=p['first_name'],
                surname=p['surname'],
                patronymic_name=p['patronymic_name'],
                doc_type=p['doc_type'],
                doc_number_hash=hash_doc_id(p['doc_number']),
                doc_number_mask_hash=hash_doc_id(AeroExpressClient.get_mask_doc_id(p['doc_number'], p['doc_type']))
            ) for p in order_data['personal_info_data']]
        else:
            passengers = [Passenger()]

        order = cls.objects.create(
            uid=order_data['order_uid'],
            status=OrderStatus.RESERVED,
            api_status=ApiOrderStatus.INIT,
            passengers=passengers,
            aeroexpress_request_data=AeroexpressRequestData(**order_data['request_data']),
            station_from_id=order_data['station_from'].id,
            station_to_id=order_data['station_to'].id,
            user_info=UserInfo(**order_data['user_info']),
            order_id=reserve_data['orderId'],
            summ=reserve_data['summ'],
            reserved_until=parser.parse(reserve_data['reservedUntil']),
            request_departure_date=parser.parse(order_data['request_data']['depart_date'])
        )

        return order
