# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import six
from mongoengine.fields import (
    IntField, StringField, DecimalField, DateField, DateTimeField, BooleanField,
    EmbeddedDocumentListField, EmbeddedDocumentField
)

from common.models.tariffs import TariffTypeCode
from travel.rasp.suburban_selling.selling.abstract_models import AbstractDocument, AbstractEmbeddedDocument


class BookData(AbstractEmbeddedDocument):
    date = StringField()  # movista date string
    station_from_express_id = StringField(required=True)
    station_to_express_id = StringField(required=True)


class Tariff(AbstractEmbeddedDocument):
    type = StringField(choices=TariffTypeCode.values())
    price = DecimalField(precision=2, min_value=0.01, required=True)
    fare_id = IntField()


@six.python_2_unicode_compatible
class MovistaTariffs(AbstractDocument):
    date = DateField(required=True)
    station_from = IntField(required=True)
    station_to = IntField(required=True)

    updated = DateTimeField(required=True)

    can_sell = BooleanField(required=True)
    book_data = EmbeddedDocumentField(BookData)
    tariffs = EmbeddedDocumentListField(Tariff)

    def __str__(self):
        return self.__repr__()

    def __repr__(self):
        return 'Tariff on {} from {} to {}'.format(self.date, self.station_from, self.station_to)

    def __hash__(self):
        return hash((self.date, self.station_from, self.station_to))

    meta = {
        'indexes': [
            {
                'fields': ['station_from', 'station_to', 'date'],
                'name': 'key_index',
                'unique': True,
            },
        ],
        'index_background': True,
    }


class MovistaStations(AbstractDocument):
    station_id = IntField(required=True)
    has_wicket = BooleanField(required=True)
    wicket_type = StringField()

    meta = {'indexes': ['station_id']}
