# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, timedelta
from typing import Dict

from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.api_clients.travel_api.providers.movista import (
    WICKET_TYPES_BY_MOVISTA_CODES, WicketDeviceType, MovistaServiceBookData
)

from travel.rasp.suburban_selling.selling.movista.models import MovistaStations
from travel.rasp.suburban_selling.selling.order.helpers import (
    ForbiddenOrderError, SuburbanServiceFactory, WrongOrderDataError
)


class WrongWicketError(Exception):
    pass


def get_movista_wicket_device(station_id):
    # type: (int) -> WicketDeviceType
    station = MovistaStations.objects.get(station_id=station_id)
    movista_device_code = station['wicket_type']

    if movista_device_code in WICKET_TYPES_BY_MOVISTA_CODES:
        return WICKET_TYPES_BY_MOVISTA_CODES[movista_device_code]

    raise WrongWicketError('Wrong type of wicket: {}, station: {}'.format(movista_device_code, station_id))


class MovistaServiceFactory(SuburbanServiceFactory):
    def raise_if_order_forbidden(self):
        # type: () -> ()
        if not conf.SUBURBAN_SELLING__MOVISTA_ORDER_ENABLED:
            raise ForbiddenOrderError(
                'Movista order is forbidden by the setting SUBURBAN_SELLING__MOVISTA_ORDER_ENABLED'
            )
        if not conf.SUBURBAN_SELLING_ENABLED:
            raise ForbiddenOrderError(
                'Movista order is forbidden by the setting SUBURBAN_SELLING_ENABLED'
            )

    def make_book_data(self, order_data):
        # type: (Dict) -> MovistaServiceBookData

        book_data = order_data['book_data']
        wicket_device = get_movista_wicket_device(order_data['station_from'].id)

        # todo: поменять на MovistaClient.dt_to_movista_date
        today = datetime.strftime(environment.now() - timedelta(hours=1), '%Y-%m-%d')
        if book_data['date'] < today:
            raise WrongOrderDataError('Order date is in the past: {}'.format(book_data['date']))

        return MovistaServiceBookData(
            book_data['date'], int(book_data['station_from_express_id']), int(book_data['station_to_express_id']),
            book_data['fare_id'], wicket_device
        )
