# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from typing import Dict, List, Optional, Iterable
from collections import namedtuple
from datetime import datetime, timedelta, time
from decimal import Decimal

from django.conf import settings
from mongoengine import Q

from common.dynamic_settings.default import conf
from common.models.tariffs import TariffTypeCode, SuburbanSellingFlow
from common.settings.utils import define_setting
from common.utils.namedtuple import namedtuple_with_defaults
from common.utils.date import MSK_TZ
from travel.rasp.library.python.api_clients.movista import MovistaClient
from travel.library.python.tracing.instrumentation import traced_function

from travel.rasp.suburban_selling.selling.tariffs.from_to_key_cache_provider import (
    FromToKeyCacheTariffsProvider, TariffFromToKey, get_stations_express_codes
)
from travel.rasp.suburban_selling.selling.movista.models import MovistaTariffs
from travel.rasp.suburban_selling.selling.tariffs.interfaces import SellingTariff, TariffKeyData, TariffKeyDataStatus
from travel.rasp.suburban_selling.selling.tariffs.tariffs_configuration import TariffsConfiguration
from travel.rasp.suburban_selling.selling.tariffs.selling_companies import SuburbanCarrierCode


log = logging.getLogger(__name__)


define_setting('MOVISTA_TARIFFS_TTL', default=3600)

define_setting('MOVISTA_TICKET_VALID_DAYS', default=1)
define_setting('MOVISTA_TICKET_VALID_HOURS_IN_LAST_DAY', default=3)


MOVISTA_FARE_PLANS = {
    'Пассажирский': TariffTypeCode.USUAL,
    'ЭКСПРЕСС': TariffTypeCode.EXPRESS,
}

MovistaTariff = namedtuple('MovistaTariff', [
    'type',     # type: str
    'price',    # type: Decimal
    'fare_id',  # type: int
])

MovistaBookData = namedtuple('MovistaTariff', [
    'date',                     # type: str
    'station_from_express_id',  # type: str
    'station_to_express_id'     # type: str
])

MovistaTariffsData = namedtuple_with_defaults(
    'MovistaTariffsData',
    [
        'can_sell',   # type: bool
        'tariffs',    # type: List[MovistaTariff]
        'book_data',  # type: MovistaBookData
        'updated',    # type: datetime
        'status'      # type: TariffKeyDataStatus
    ],
    defaults={'updated': None}
)


class MovistaTariffsProvider(FromToKeyCacheTariffsProvider):
    def __init__(self, movista_client, tariffs_ttl, tariffs_configuration):
        # type: (MovistaClient, int, TariffsConfiguration) -> None
        super(FromToKeyCacheTariffsProvider, self).__init__(tariffs_configuration)

        self.tariffs_ttl = tariffs_ttl
        self.movista_client = movista_client

    def get_selling_flows(self):
        return {SuburbanSellingFlow.VALIDATOR}

    def check_get_tariffs_enabled(self):
        # type: () -> bool
        return conf.SUBURBAN_SELLING__MOVISTA_TARIFFS_ENABLED

    @traced_function
    def get_tariffs_data_from_cache(self, from_to_keys):
        # type: (Iterable[TariffFromToKey]) -> Dict[TariffFromToKey, MovistaTariffsData]

        if not from_to_keys:
            return {}

        tariffs_query = Q()
        for key in from_to_keys:
            tariffs_query |= Q(
                date=key.date,
                station_from=key.station_from,
                station_to=key.station_to,
                updated__gte=datetime.utcnow() - timedelta(seconds=self.tariffs_ttl)
            )

        keys_with_tariffs = {}
        for tariffs_doc in MovistaTariffs.objects.filter(tariffs_query):
            tariff_key = TariffFromToKey(
                date=tariffs_doc.date,
                station_from=tariffs_doc.station_from,
                station_to=tariffs_doc.station_to,
            )
            keys_with_tariffs[tariff_key] = MovistaTariffsData(
                can_sell=tariffs_doc.can_sell,
                book_data=MovistaBookData(
                    date=tariffs_doc.book_data.date,
                    station_from_express_id=tariffs_doc.book_data.station_from_express_id,
                    station_to_express_id=tariffs_doc.book_data.station_to_express_id,
                ),
                tariffs=[
                    MovistaTariff(
                        type=tariff.type,
                        price=tariff.price,
                        fare_id=tariff.fare_id,
                    )
                    for tariff in tariffs_doc['tariffs']
                ],
                updated=tariffs_doc.updated,
                status=TariffKeyDataStatus.ACTUAL
            )

        return keys_with_tariffs

    @traced_function
    def get_tariffs_data_from_provider(self, from_to_key):
        # type: (TariffFromToKey) -> Optional[MovistaTariffsData]

        if not conf.SUBURBAN_SELLING__MOVISTA_API_CALL_ENABLED:
            return

        express_ids = get_stations_express_codes(from_to_key)
        if not express_ids:
            return
        station_from_express_id, station_to_express_id = express_ids

        try:
            fares = self.movista_client.fares(from_to_key.date, station_from_express_id, station_to_express_id)
            tariffs = []
            for fare in fares['fares']:
                tariff_type = MOVISTA_FARE_PLANS.get(fare['farePlan'])
                if tariff_type:
                    tariffs.append(MovistaTariff(
                        type=tariff_type,
                        price=self.movista_client.movista_price_to_decimal(fare['price']),
                        fare_id=fare['fareId'],
                    ))
                else:
                    log.exception('Unknown Movista fare plan %s in: %s', fare['farePlan'], fares)

            can_sell = fares['sale']

        except Exception:
            log.exception('Can\'t get tariffs for %s', from_to_key)
            return

        return MovistaTariffsData(
            can_sell=can_sell,
            tariffs=tariffs,
            book_data=MovistaBookData(
                date=self.movista_client.dt_to_movista_date(from_to_key.date),
                station_from_express_id=station_from_express_id,
                station_to_express_id=station_to_express_id,
            ),
            updated=datetime.utcnow(),
        )

    @traced_function
    def save_tariffs_data_to_cache(self, from_to_key, tariffs_data_for_key):
        # type: (TariffFromToKey, MovistaTariffsData) -> None

        MovistaTariffs.objects(
            date=from_to_key.date,
            station_from=from_to_key.station_from,
            station_to=from_to_key.station_to,
        ).update_one(
            can_sell=tariffs_data_for_key.can_sell,
            book_data={
                'date': tariffs_data_for_key.book_data.date,
                'station_from_express_id': tariffs_data_for_key.book_data.station_from_express_id,
                'station_to_express_id': tariffs_data_for_key.book_data.station_to_express_id,
            },
            tariffs=[
                {
                    'type': tariff.type,
                    'price': tariff.price,
                    'fare_id': tariff.fare_id,
                }
                for tariff in tariffs_data_for_key.tariffs
            ],
            updated=datetime.utcnow(),
            upsert=True
        )

    @traced_function
    def make_selling_tariffs_from_tariffs_data(self, tariffs_data_for_all_keys):
        # type: (Dict[TariffFromToKey, MovistaTariffsData]) -> Dict[TariffFromToKey, TariffKeyData]

        data_by_from_to_keys = {}  # type: Dict[TariffFromToKey, TariffKeyData]
        for from_to_key, tariffs_data in tariffs_data_for_all_keys.items():
            data_by_from_to_keys[from_to_key] = TariffKeyData([], tariffs_data.status)
            if not tariffs_data or not tariffs_data.can_sell:
                continue

            selling_tariffs = self.tariffs_configuration.tariff_types_by_carriers_and_codes[SuburbanCarrierCode.CPPK]
            for tariff in tariffs_data.tariffs:
                tariff_date = self.movista_client.date_from_movista_date(tariffs_data.book_data.date)
                valid_from = MSK_TZ.localize(datetime.combine(tariff_date, time(0)))
                valid_until = MSK_TZ.localize(
                    datetime.combine(
                        tariff_date + timedelta(days=settings.MOVISTA_TICKET_VALID_DAYS),
                        time(settings.MOVISTA_TICKET_VALID_HOURS_IN_LAST_DAY)
                    )
                )
                selling_tariff = SellingTariff(
                    partner=SuburbanCarrierCode.CPPK,
                    tariff_type=tariff.type,
                    name=tariff.type,
                    description=selling_tariffs[tariff.type].description,
                    price=round(tariff.price, 2),
                    max_days=None,
                    valid_from=valid_from,
                    valid_until=valid_until,
                    book_data=dict(
                        date=tariffs_data.book_data.date,
                        station_from_express_id=tariffs_data.book_data.station_from_express_id,
                        station_to_express_id=tariffs_data.book_data.station_to_express_id,
                        fare_id=tariff.fare_id,
                    ),
                    selling_flow=SuburbanSellingFlow.VALIDATOR
                )
                data_by_from_to_keys[from_to_key].tariffs.append(selling_tariff)

        return data_by_from_to_keys
