# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
import logging
import base64
from typing import AnyStr, Optional

from django.http import HttpRequest

from travel.rasp.library.python.common23.data_api.juggler.juggler import CheckError, CRIT, get_events, send_events
from common.data_api.redir.instance import redir_client

from travel.proto.suburban.cpa_label_pb2 import TSuburbanCpaLabel


log = logging.getLogger(__name__)


def _add_label_property(label, create_order_request, property_name, header_name):
    header_value = create_order_request.META.get(header_name)
    if header_value:
        setattr(label, property_name, header_value)


def make_serialized_label_for_redir(create_order_request, passport_uid):
    # type: (HttpRequest, Optional[AnyStr]) -> AnyStr

    label = TSuburbanCpaLabel()
    _add_label_property(label, create_order_request, 'Uuid', 'HTTP_X_YA_UUID')
    _add_label_property(label, create_order_request, 'DeviceId', 'HTTP_X_YA_DEVICE_ID')

    if passport_uid:
        label.PassportUid = passport_uid

    _add_label_property(label, create_order_request, 'TestBuckets', 'HTTP_X_TEST_BUCKETS')

    user_agent = create_order_request.META.get('HTTP_USER_AGENT')
    if user_agent:
        try:
            # App: Suburban {VERSION_NAME}({VERSION_CODE}); OS: Android ${VERSION}; ...
            agent_parts = re.split('[:;]? ', user_agent)
            version_parts = re.split('[()]', agent_parts[2])

            label.OsType = agent_parts[4]
            label.OsVersion = agent_parts[5]
            label.AppVersion = version_parts[0]
            label.AppCodeVersion = version_parts[1]
        except:
            log.exception('Header User-Agent has wrong format: {}'.format(user_agent))

    _add_label_property(label, create_order_request, 'Device', 'HTTP_X_DEVICE')
    _add_label_property(label, create_order_request, 'Ip', 'HTTP_X_REAL_IP')
    _add_label_property(label, create_order_request, 'CpaExtraData', 'HTTP_X_CPA_EXTRA_DATA')

    return base64.b64encode(label.SerializeToString())


def has_redir_errors():
    raise CheckError(CRIT, 'Redir call error')


def get_label_hash_from_redir(create_order_request, passport_uid):
    # type: (HttpRequest, AnyStr) -> AnyStr
    """
    Передает данные пользователя из заказа в редиректилку
    :return: hash лейбла, возвращенный редиректилкой
    """
    try:
        cpa_label = make_serialized_label_for_redir(create_order_request, passport_uid)
        cpa_label_hash = redir_client.suburban_label_to_hash(cpa_label)

    except Exception:
        cpa_label_hash = 'ERROR_REDIR_BAD_RESPONSE'

        tags = ['redir', 'suburban-selling']
        checks_events = get_events([
            ('suburban-selling-redir-errors', has_redir_errors, tags),
        ])
        send_events(checks_events)

    return cpa_label_hash
