# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import json
from typing import Dict

import requests
from django.http import JsonResponse, HttpRequest
from django.views.decorators.http import require_http_methods

from common.data_api.travel_api.instance import travel_api
from common.models.tariffs import SuburbanTariffProvider

from travel.rasp.library.python.api_clients.travel_api.client import (
    TravelApiUserIdent, get_travel_api_order_state, OrderState
)
from travel.rasp.library.python.api_clients.travel_api.create_order_info import TravelApiCreateOrderInfo

from travel.rasp.suburban_selling.selling.aeroexpress.order import AeroexpressServiceFactory
from travel.rasp.suburban_selling.selling.im.order import ImServiceFactory
from travel.rasp.suburban_selling.selling.movista.order import MovistaServiceFactory
from travel.rasp.suburban_selling.selling.order.cpa import get_label_hash_from_redir
from travel.rasp.suburban_selling.selling.order.helpers import (
    make_and_check_user_ident, ForbiddenOrderError, WrongOrderDataError
)
from travel.rasp.suburban_selling.selling.order.request_serialization import CreateOrderRequestSchema


log = logging.getLogger(__name__)

CREATE_ORDER_V1 = 1
CREATE_ORDER_V2 = 2
CREATE_ORDER_V3 = 3


class TravelApiTestTokens(object):
    def __init__(self, request):
        self.test_context_token = request.META.get('HTTP_X_TEST_CONTEXT_TOKEN')
        self.payment_test_context_token = request.META.get('HTTP_X_PAYMENT_TEST_CONTEXT_TOKEN')


def create_travel_api_order(order_data, user_ident, cpa_label_hash, test_tokens):
    # type: (Dict, TravelApiUserIdent, str, TravelApiTestTokens) -> (str, str)
    """Создание заказа билета через TravelAPI"""

    provider = order_data['provider']
    if provider == SuburbanTariffProvider.MOVISTA:
        service_factory = MovistaServiceFactory()
    elif provider == SuburbanTariffProvider.IM:
        service_factory = ImServiceFactory()
    elif provider == SuburbanTariffProvider.AEROEXPRESS:
        service_factory = AeroexpressServiceFactory()
    else:
        raise ForbiddenOrderError('Unknown provider {}'.format(provider))

    service = service_factory.make_service(order_data, test_tokens.test_context_token)
    log.info('Create order: {}, test_token: {}'.format(
        json.dumps(service.request_json), test_tokens.payment_test_context_token
    ))

    user_info = order_data['user_info']
    travel_api_order_info = TravelApiCreateOrderInfo(
        label=cpa_label_hash,
        phone=user_info['phone'],
        email=user_info['email'],
        geo_id=user_info['region_id'],
        ip=user_info['ip'],
        suburban_services=[service],
        payment_test_context_token=test_tokens.payment_test_context_token
    )

    api_response = travel_api.create_order(user_ident, travel_api_order_info.request_json)
    state = get_travel_api_order_state(api_response['state'], OrderState.NEW)

    log.info('Order {} was created successfully'.format(api_response['id']))

    return api_response['id'], state


def _process_empty_carrier(order_data, version):
    if order_data['provider'] == SuburbanTariffProvider.AEROEXPRESS:
        order_data.setdefault('partner', 'aeroexpress')
    elif version == CREATE_ORDER_V2 and order_data['provider'] == SuburbanTariffProvider.MOVISTA:
        order_data.setdefault('partner', 'cppk')
    elif not order_data.get('partner', ''):
        raise ForbiddenOrderError('Unavailable empty carrier')


@require_http_methods(['POST'])
def create_order(request):
    # type: (HttpRequest) -> JsonResponse

    try:
        request_params = request.POST.dict()
        version = int(request_params.get('version', '1'))
        if version == CREATE_ORDER_V1:
            log.error('Selling version 1 is not supported')
            return JsonResponse({'status': 'failed', 'error': 'Selling version 1 is not supported'}, status=400)

        log.info('Parsing create_order request')
        order_data, errors = CreateOrderRequestSchema().load(request_params)

        if errors:
            log.error('Bad request. {}'.format(errors))
            return JsonResponse({'status': 'failed', 'errors': errors}, status=400)

        _process_empty_carrier(order_data, version)

    except Exception:
        log.exception('create_order bad request')
        return JsonResponse({'status': 'failed'}, status=400)

    user_ident, error_response = make_and_check_user_ident(request)
    if error_response:
        return error_response

    cpa_label_hash = get_label_hash_from_redir(request, user_ident.blackbox_user_uid)
    try:
        test_tokens = TravelApiTestTokens(request)
        order_uid, order_status = create_travel_api_order(order_data, user_ident, cpa_label_hash, test_tokens)

        return JsonResponse({
            'order_link': 'https://{}/order_info/?uid={}'.format(request.get_host(), order_uid),
            'status': order_status,
            'uid': order_uid
        })

    except ForbiddenOrderError as ex:
        return JsonResponse({'status': 'failed', 'error': ex.message}, status=403)

    except WrongOrderDataError as ex:
        return JsonResponse({'status': 'wrong order data', 'error': ex.message}, status=404)

    except requests.HTTPError as ex:
        if ex.response.status_code in {401, 403, 404}:
            return JsonResponse({'status': 'failed', 'error': 'Client error'}, status=ex.response.status_code)
