# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from typing import Optional, Dict, AnyStr

from django.http import HttpRequest, JsonResponse

from common.utils.blackbox_wrapper import SessionInvalid
from travel.rasp.library.python.api_clients.travel_api.client import TravelApiUserIdent
from travel.rasp.library.python.api_clients.travel_api.create_order_info import SuburbanService, SuburbanServiceBookData

from travel.rasp.suburban_selling.selling.views.utils import get_user_from_blackbox


log = logging.getLogger(__name__)


class UserUnauthorizedError(Exception):
    pass


class UserForbiddenError(Exception):
    pass


def make_user_ident(request):
    # type: (HttpRequest) -> Optional[TravelApiUserIdent]

    uuid = request.META.get('HTTP_X_YA_UUID')  # мобилки передают сюда uuid инсталляции приложения (берем у аппметрики)
    device_id = request.META.get('HTTP_X_YA_DEVICE_ID')
    user_agent = request.META.get('HTTP_USER_AGENT')
    user_ip = request.META.get('HTTP_X_REAL_IP')

    if not uuid:
        log.error('Empty X-Ya-Uuid header in request {}'.format(request.path))
        raise UserUnauthorizedError('Empty X-Ya-Uuid header in request')

    try:
        user_uid, user_ticket = get_user_from_blackbox(request)
    except SessionInvalid:
        raise UserForbiddenError('Can not get an user ticket from blackbox')

    return TravelApiUserIdent(
        session_id=uuid,
        yandex_uid=uuid,
        device_id=device_id,
        user_ticket=user_ticket,
        blackbox_user_uid=user_uid,
        user_agent=user_agent,
        user_ip=user_ip
    )


def make_and_check_user_ident(request):
    # type: (HttpRequest) -> (Optional[TravelApiUserIdent], Optional[JsonResponse])
    try:
        user_ident = make_user_ident(request)
        return user_ident, None
    except UserUnauthorizedError as ex:
        return None, JsonResponse({'status': 'failed', 'error': ex.message}, status=401)
    except UserForbiddenError as ex:
        return None, JsonResponse({'status': 'failed', 'error': ex.message}, status=403)


class ForbiddenOrderError(Exception):
    pass


class WrongOrderDataError(Exception):
    pass


class SuburbanServiceFactory(object):
    def raise_if_order_forbidden(self):
        # type: () -> ()
        raise NotImplementedError

    def make_book_data(self, order_data):
        # type: (Dict) -> SuburbanServiceBookData
        raise NotImplementedError

    def make_service(self, order_data, test_context_token=None):
        # type: (Dict, Optional[AnyStr]) -> SuburbanService

        self.raise_if_order_forbidden()

        station_from = order_data['station_from']
        station_to = order_data['station_to']
        return SuburbanService(
            provider=order_data['provider'],
            station_from_id=station_from.id,
            station_to_id=station_to.id,
            price=order_data['price'],
            carrier_partner=order_data.get('partner'),
            book_data=self.make_book_data(order_data),
            test_context_token=test_context_token
        )
