# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from typing import Dict, Optional, AnyStr

import requests
from dateutil import parser
from django.conf import settings
from django.http import HttpResponse, JsonResponse, HttpRequest
from django.views.decorators.http import require_http_methods
from marshmallow import Schema, fields

from common.data_api.travel_api.instance import travel_api
from common.models.tariffs import SuburbanSellingFlow
from common.settings.configuration import Configuration
from common.settings.utils import define_setting

from travel.rasp.library.python.api_clients.travel_api.client import (
    TravelApiUserIdent, get_travel_api_order_state, OrderState
)
from travel.rasp.suburban_selling.selling.order.helpers import make_and_check_user_ident
from travel.rasp.suburban_selling.selling.aeroexpress.models import AeroexMenu


log = logging.getLogger(__name__)

ORDER_INFO_V1 = 1
ORDER_INFO_V2 = 2
ORDER_INFO_V3 = 3


define_setting('TRUST_RETURN_PATH', {
    Configuration.PRODUCTION: 'https://export.rasp.yandex.net/static/payment_completed/',
    Configuration.TESTING: 'https://testing.export.rasp.yandex.net/static/payment_completed/'
}, default='https://testing.export.rasp.yandex.net/static/payment_completed/')


class OrderInfoCommand(object):
    START_PAYMENT = 'start_payment'
    GET_PAYMENT_DATA = 'get_payment_data'
    GET_TICKET = 'get_ticket'

    ALL = {START_PAYMENT, GET_PAYMENT_DATA, GET_TICKET}


def get_travel_api_order_info(user_ident, order_id, command=None, version=ORDER_INFO_V3):
    # type: (TravelApiUserIdent, AnyStr, Optional[AnyStr], int) -> Dict
    """
    Информация о заказе от Travel API и запуск команд управления заказом (пока только start_payment)
    """
    state = travel_api.get_order_state(user_ident, order_id)['state']
    log.info('Order: {}, state from Travel API: {}'.format(order_id, state))

    if command == OrderInfoCommand.START_PAYMENT and state in {OrderState.RESERVED, OrderState.PAYMENT_FAILED}:
        # Запуск оплаты
        return start_payment_order_info(user_ident, order_id, state)

    if command == OrderInfoCommand.GET_PAYMENT_DATA and state == OrderState.WAITING_PAYMENT:
        # Получение данных для открытия Web View
        return get_payment_data_order_info(user_ident, order_id, state)

    if command == OrderInfoCommand.GET_TICKET and state == OrderState.CONFIRMED:
        # Получение билета
        return get_ticket_order_info(user_ident, order_id, version)

    # Во всех остальных случаях просто возвращаем текущее состояние
    # Специфические для бэкенда внутренние состояния превращаем в IN_PROGRESS
    return {'status': get_travel_api_order_state(state, OrderState.IN_PROGRESS)}


def start_payment_order_info(user_ident, order_id, state):
    # type: (TravelApiUserIdent, AnyStr, AnyStr) -> Dict

    payment_request_content = {
        'order_id': order_id,
        'return_url': settings.TRUST_RETURN_PATH,
        'source': 'mobile'
    }

    try:
        travel_api.start_payment(user_ident, payment_request_content)
    except requests.HTTPError as ex:
        if ex.response.status_code != 409:
            raise

    log.info('Order: {}. Payment started'.format(order_id))

    return {'status': state}


def get_payment_data_order_info(user_ident, order_id, state):
    # type: (TravelApiUserIdent, AnyStr, AnyStr) -> Dict

    api_response = travel_api.get_order(user_ident, order_id)
    payment_url = api_response['payment']['current']['payment_url']
    purchase_token = api_response['payment']['current'].get('purchase_token', '')
    log.info('Order: {}. payment data received'.format(order_id))

    return {
        'status': state,
        'payment_url': payment_url,
        'purchase_token': purchase_token,
    }


def get_ticket_order_info(user_ident, order_id, version):
    # type: (TravelApiUserIdent, AnyStr, int) -> Dict

    api_response = travel_api.get_order(user_ident, order_id)
    log.info('Order: {}. Ticket received'.format(order_id))

    if api_response['services'][0]['suburban_info']['flow'] != SuburbanSellingFlow.AEROEXPRESS:
        return _get_order_info_from_api_response(api_response, version)
    else:
        return _get_aeroexpress_info_from_api_response(api_response)


def _get_order_info_from_api_response(api_response, version):
    # type: (Dict, int) -> Dict

    status = get_travel_api_order_state(api_response['state'], OrderState.IN_PROGRESS)

    service = api_response['services'][0]['suburban_info']
    ticket_number = service['ticket_number']

    result = {
        'status': status,
        'price': _get_price(api_response),
        'ticket_body': service['ticket_body'],
        'flow': service['flow'],
        'barcode_preset': service['barcode_preset']
    }

    if service['flow'] == SuburbanSellingFlow.VALIDATOR:
        result['wicket'] = service['wicket']
        if version == ORDER_INFO_V2:
            ticket_number = int(ticket_number)

    result['ticket_number'] = ticket_number

    return result


def _get_aeroexpress_info_from_api_response(api_response):
    # type: (Dict) -> Dict

    suburban_service = api_response['services'][0]['suburban_info']
    service_ae_info = suburban_service['aeroexpress_info']

    try:
        aero = AeroexMenu.objects.get(menu_id=service_ae_info['menu_id'])
    except AeroexMenu.DoesNotExist:
        trip_count, description = 1, ''
    else:
        trip_count, description = aero.get_trip_count(), aero.get_description()

    passenger = {
        'first_name': None,
        'surname': None,
        'patronymic_name': None,
        'ticket': {
            'ticket_url': '{}&type=html'.format(service_ae_info['ticket_url']),
            'code_url': '{}&type=qr'.format(service_ae_info['ticket_url']),
            'price': _get_price(api_response),
            'tariff': service_ae_info['tariff'],
            'route': service_ae_info['st_depart'],
            'trip_date': service_ae_info['trip_date'],
            'dead_date': service_ae_info['valid_until'],
            'trip_count': trip_count,
            'description': description
        }
    }

    result = {
        'passengers': [passenger],
        'status': get_travel_api_order_state(api_response['state'], OrderState.IN_PROGRESS),
        'price': _get_price(api_response),
        'ticket_body': suburban_service['ticket_body'],
        'ticket_number': int(suburban_service['ticket_number']),
        'flow': suburban_service['flow'],
        'barcode_preset': suburban_service['barcode_preset'],
        'email': api_response['contact_info']['email'],
        'phone': api_response['contact_info']['phone'],
        'payment_url': api_response['payment']['current']['payment_url'],
        'purchase_token': api_response['payment']['current'].get('purchase_token', ''),
        'create_dt': parser.parse(api_response['serviced_at']).isoformat(),

    }
    return result


def _get_price(api_response):
    # type: (Dict) -> float
    return (api_response['order_price_info'].get('price') or {}).get('value')


class OrderInfoRequestSchema(Schema):
    uid = fields.String(required=True)
    version = fields.Int(missing=ORDER_INFO_V1)
    command = fields.String(missing=None)
    is_aeroexpress = fields.Boolean(missing=False)


@require_http_methods(['GET'])
def order_info(request):
    # type: (HttpRequest) -> HttpResponse
    """Получение состояния и информации о заказе, и запуск команд управления заказом"""

    request_params, errors = OrderInfoRequestSchema().load(request.GET)
    if errors:
        log.error('Bad request. {}'.format(errors))
        return JsonResponse({'status': 'failed', 'errors': errors}, status=400)

    command = request_params['command']
    if command and command not in OrderInfoCommand.ALL:
        return JsonResponse({'status': 'failed', 'error': 'Unknown command "{}"'.format(command)}, status=400)

    version = request_params['version']
    if version == ORDER_INFO_V1:
        log.exception('Selling version 1 is not supported')
        return JsonResponse({'status': 'failed', 'error': 'Selling version 1 is not supported'}, status=400)

    user_ident, error_response = make_and_check_user_ident(request)
    if error_response:
        return error_response

    order_dict = {}
    try:
        order_dict = get_travel_api_order_info(user_ident, request_params['uid'], command, version)

    except requests.HTTPError as ex:
        if ex.response.status_code in {401, 403, 404}:
            return JsonResponse(
                {'status': 'failed', 'error': 'Client error in {}'.format(ex.request.url)},
                status=ex.response.status_code
            )

    return HttpResponse(json.dumps(order_dict, ensure_ascii=False), content_type='application/json; charset=utf-8')
