# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from typing import List

import requests
from django.http import JsonResponse, HttpRequest
from django.views.decorators.http import require_http_methods
from marshmallow import Schema, fields

from common.data_api.travel_api.instance import travel_api
from travel.rasp.library.python.api_clients.travel_api.client import (
    TravelApiUserIdent, get_travel_api_order_state, OrderState
)

from travel.rasp.suburban_selling.selling.order.helpers import make_and_check_user_ident


log = logging.getLogger(__name__)


class OrderSchema(Schema):
    uid = fields.String(required=True)
    provider = fields.String(required=True)


class OrdersListInfoRequestSchema(Schema):
    orders = fields.Nested(OrderSchema, required=True, many=True)


def _get_travel_api_orders_data(user_ident, travel_api_orders_uids):
    # type: (TravelApiUserIdent, List[str]) -> List[dict]

    request_data = {'order_ids': travel_api_orders_uids}
    travel_api_response = travel_api.get_order_state_batch(user_ident, request_data)
    return [
        {
            'uid': order['order_id'],
            'status': get_travel_api_order_state(order['state'], OrderState.IN_PROGRESS)
        } for order in travel_api_response['orders']
    ]


@require_http_methods(['POST'])
def orders_list_info(request):
    # type: (HttpRequest) -> JsonResponse

    request_json = json.loads(request.body)
    request_dict, errors = OrdersListInfoRequestSchema().load(request_json)

    if errors:
        log.error('Bad request. {}'.format(errors))
        return JsonResponse({'status': 'failed', 'errors': errors}, status=400)

    user_ident, error_response = make_and_check_user_ident(request)
    if error_response:
        return error_response

    orders_data = []
    try:
        travel_api_orders_uids = [order['uid'] for order in request_dict['orders']]
        orders_data = _get_travel_api_orders_data(user_ident, travel_api_orders_uids)
    except requests.HTTPError as ex:
        if ex.response.status_code in {401, 403, 404}:
            return JsonResponse(
                {'status': 'failed', 'error': 'Client error in {}'.format(ex.request.url)},
                status=ex.response.status_code
            )

    return JsonResponse({'orders': orders_data})
