# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import re

from dateutil import parser
from marshmallow import Schema, fields, pre_load, post_load, validates_schema, ValidationError
from django.conf import settings

from common.settings.utils import define_setting
from common.models.geo import Station
from common.utils.date import MSK_TZ
from common.models.tariffs import SuburbanTariffProvider


log = logging.getLogger(__name__)


# Вместо телефона пользователя в оркестратор передаем телефон поддержки
define_setting('SELLING_SUPPORT_TELEPHONE', default='1')


EMAIL_RE = re.compile(r'(^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$)')


def validate_email(value):
    if not EMAIL_RE.match(value):
        raise Exception('Невалидный email {}'.format(value))


class DocField(fields.String):
    symbols = [' ']

    def _deserialize(self, value, attr, data):
        doc_number = super(DocField, self)._deserialize(value, attr, data)
        try:
            return re.sub('|'.join(self.symbols), '', doc_number.strip())
        except Exception:
            self.fail('Не смогли десериализовать')


class PhoneField(fields.String):
    symbols = [' ', r'\(', r'\)', '-', r'\+']

    def _deserialize(self, value, attr, data):
        phone_number = super(PhoneField, self)._deserialize(value, attr, data)
        try:
            return re.sub('|'.join(self.symbols), '', phone_number.strip())
        except Exception:
            self.fail('Не смогли десериализовать')


class StripStringField(fields.String):
    def _deserialize(self, value, attr, data):
        value = super(StripStringField, self)._deserialize(value, attr, data)
        return value.strip()


class UserInfoSchema(Schema):
    email = fields.Email(required=True, validate=validate_email)
    phone = PhoneField(missing=settings.SELLING_SUPPORT_TELEPHONE)
    region_id = fields.Int(missing=0)
    ip = fields.String(missing=None)


class StationField(fields.Integer):
    def _deserialize(self, value, attr, data):
        station_id = super(StationField, self)._deserialize(value, attr, data)
        try:
            return Station.objects.get(id=int(station_id))
        except Station.DoesNotExist:
            self.fail('Станция не найдена, id={}'.format(station_id))


def _change_departure_date_in_data(data):
    departure_date = data.get('departure_date')
    if departure_date:
        departure_date = parser.parse(departure_date.replace(' ', '+')).replace(tzinfo=None)
        data['departure_date'] = MSK_TZ.localize(departure_date).isoformat()


class CreateOrderRequestSchema(Schema):
    version = fields.Integer()
    provider = fields.String(required=True)
    partner = fields.String()
    station_from = StationField(required=True)
    station_to = StationField(required=True)
    departure_date = fields.String(required=True)
    price = fields.Float(required=True)

    book_data = fields.String(required=True)
    user_info = fields.Nested(UserInfoSchema, required=True)

    @pre_load
    def params_to_json(self, data):
        user_info = data.get('user_info')
        if user_info:
            data['user_info'] = json.loads(user_info)

        _change_departure_date_in_data(data)
        return data

    @post_load
    def load_book_data(self, data):
        data['book_data'] = json.loads(data['book_data'])

    @validates_schema(skip_on_field_errors=True)
    def validate_params(self, data):
        if data['provider'] not in SuburbanTariffProvider.ALL:
            raise ValidationError('Unknown tariffs provider', ['provider'])
