# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from marshmallow import Schema, fields
from django.http import JsonResponse, HttpRequest
from django.views.decorators.http import require_http_methods

from common.data_api.travel_api.instance import travel_api
from common.data_api.movista.instance import movista_client
from travel.rasp.library.python.common23.date import environment

from travel.rasp.suburban_selling.selling.order.helpers import (
    make_user_ident, UserUnauthorizedError, UserForbiddenError
)


log = logging.getLogger(__name__)


class UserActionEventRequestSchema(Schema):
    uid = fields.String(required=True)
    is_success = fields.Boolean(required=True)
    qr_body = fields.String()


@require_http_methods(['POST'])
def user_action_event(request):
    # type: (HttpRequest) -> JsonResponse

    request_params, errors = UserActionEventRequestSchema().load(request.POST.dict())
    if errors:
        log.error('Bad request. {}'.format(errors))
        return JsonResponse({'status': 'failed', 'errors': errors}, status=400)

    try:
        user_ident = make_user_ident(request)
    except UserUnauthorizedError as ex:
        return JsonResponse({'status': 'failed', 'error': ex.message}, status=401)
    except UserForbiddenError as ex:
        return JsonResponse({'status': 'failed', 'error': ex.message}, status=403)

    order = travel_api.get_order(user_ident, request_params['uid'])
    service = order['services'][0]['suburban_info']

    log.info('Order activation in Movista, order_id={}, is_success={}'.format(
        service['partner_order_id'], request_params['is_success']
    ))

    api_response = movista_client.activation(
        order_id=service['partner_order_id'],
        dt=environment.now(),
        is_success=request_params['is_success'],
        qr_body=request_params['qr_body'],
        ticket_body=service['ticket_body']
    )

    log.info('Order activation in Movista is completed, activation_id={}'.format(api_response['activationId']))

    return JsonResponse({
        'result': api_response['result'],
        'activation_id': api_response['activationId'],
    })
