# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from typing import List, Dict

from django.conf import settings

from common.dynamic_settings.default import conf
from common.data_api.movista.instance import movista_client
from common.models.tariffs import SuburbanTariffProvider
from common.settings.configuration import Configuration

from travel.rasp.suburban_selling.selling.aeroexpress.tariffs import AeroexTariffsProvider
from travel.rasp.suburban_selling.selling.movista.tariffs import MovistaTariffsProvider
from travel.rasp.suburban_selling.selling.im.tariffs import ImTariffsProvider
from travel.rasp.suburban_selling.selling.tariffs.tariffs_getter import TariffsGetter
from travel.rasp.suburban_selling.selling.tariffs.tariffs_configuration import TariffsConfiguration
from travel.rasp.suburban_selling.selling.tariffs.interfaces import TariffKey  # noqa
from travel.rasp.suburban_selling.selling.tariffs.selling_companies import (
    SUBURBAN_CARRIERS_BY_COMPANIES_ID, SUBURBAN_CARRIERS_BY_CODES
)


log = logging.getLogger(__name__)


def _carrier_selling_enabled(carrier):
    env_type = settings.YANDEX_ENVIRONMENT_TYPE

    if carrier.code == 'mtppk' and not conf.SUBURBAN_MTPPK_SELLING_ENABLED:
        return False
    if carrier.code == 'szppk' and not conf.SUBURBAN_SZPPK_SELLING_ENABLED:
        return False
    if carrier.code == 'bashppk' and not conf.SUBURBAN_BASHPPK_SELLING_ENABLED:
        return False
    if carrier.code == 'sodruzhestvo' and not conf.SUBURBAN_SODRUZHESTVO_SELLING_ENABLED:
        return False

    if carrier.selling_enabled_prod and env_type in {Configuration.PRODUCTION, Configuration.STRESS}:
        return True
    if carrier.selling_enabled_testing and env_type in {Configuration.TESTING, Configuration.DEVELOPMENT}:
        return True
    return False


def get_tariffs_for_tariff_keys(tariff_keys, selling_flows, barcode_presets, tariff_type_ids_by_numbers):
    # type: (List[TariffKey], List[str], List[str], Dict[str, int]) -> Dict

    tariffs_configuration = TariffsConfiguration(
        carriers_by_codes={
            code: carrier
            for code, carrier in SUBURBAN_CARRIERS_BY_CODES.items()
            if _carrier_selling_enabled(carrier)
        },
        carrier_codes_by_company_id=SUBURBAN_CARRIERS_BY_COMPANIES_ID
    )
    tariffs_getter = TariffsGetter(
        providers={
            SuburbanTariffProvider.MOVISTA: MovistaTariffsProvider(
                movista_client=movista_client,
                tariffs_ttl=settings.MOVISTA_TARIFFS_TTL,
                tariffs_configuration=tariffs_configuration
            ),
            SuburbanTariffProvider.IM: ImTariffsProvider(
                tariff_type_ids_by_numbers=tariff_type_ids_by_numbers,
                tariffs_configuration=tariffs_configuration
            ),
            SuburbanTariffProvider.AEROEXPRESS: AeroexTariffsProvider(
                tariffs_configuration=tariffs_configuration
            )
        },
        selling_flows=selling_flows,
        barcode_presets=barcode_presets,
        tariffs_configuration=tariffs_configuration
    )

    return tariffs_getter.get_tariffs(tariff_keys)
