# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import namedtuple
from datetime import datetime, timedelta
from typing import Iterable, Dict, Any, AnyStr, Optional, List

from django.conf import settings

from common.models.geo import Station
from common.settings.utils import define_setting
from travel.library.python.tracing.instrumentation import traced_function

from travel.rasp.suburban_selling.selling.tariffs.interfaces import (
    TariffKeyData, TariffKey, TariffsProvider, TariffKeyDataStatus
)
from travel.rasp.suburban_selling.selling.tariffs.selling_companies import SUBURBAN_CARRIERS_BY_COMPANIES_ID


log = logging.getLogger(__name__)

define_setting('MAX_TARIFFS_UPDATE_TIME', default=600)


# Ключ для получения тарифов только по станциям и дате
TariffFromToKey = namedtuple(
    'TariffFromToKey',
    [
        'date',          # type: datetime.date
        'station_from',  # type: int
        'station_to'     # type: int
    ]
)


def get_tariff_from_to_key(tariff_key):
    # type: (TariffKey) -> TariffFromToKey

    return TariffFromToKey(
        date=tariff_key.date,
        station_from=tariff_key.station_from,
        station_to=tariff_key.station_to
    )


class CantFindStation(Exception):
    pass


def _get_station_express_id(station):
    # type: (Station) -> AnyStr
    express_id = station.express_id
    if not express_id:
        raise CantFindStation('Station {} ({}) has no express_id'.format(station, station.id))
    return express_id


def get_stations_express_codes(from_to_key):
    # type: (TariffFromToKey) -> (AnyStr, AnyStr)
    """Возвращает экспресс-коды для станций из ключа"""

    try:
        station_from = Station.objects.get(id=from_to_key.station_from)
        station_to = Station.objects.get(id=from_to_key.station_to)
        station_from_express_id = _get_station_express_id(station_from)
        station_to_express_id = _get_station_express_id(station_to)
        return station_from_express_id, station_to_express_id

    except (CantFindStation, Station.DoesNotExist):
        log.debug('Can\'t parse stations from %s', from_to_key)
        return


class BaseFromToKeyTariffsData(object):
    """Базовый класс для наборов тарифов по from-to ключу"""
    def __init__(self, tariffs_ttl, empty_tariffs_ttl=None, updated=None, update_started=None):
        # type: (int, Optional[int], Optional[datetime], Optional[datetime]) -> None
        self.tariffs = []

        self.updated = updated
        self.update_started = update_started
        self.tariffs_ttl = tariffs_ttl
        self.empty_tariffs_ttl = empty_tariffs_ttl or tariffs_ttl

        if not updated:
            self.status = TariffKeyDataStatus.NO_DATA
        else:
            if datetime.utcnow() - updated > timedelta(seconds=tariffs_ttl):
                self.status = TariffKeyDataStatus.OLD
            else:
                self.status = TariffKeyDataStatus.ACTUAL

    @property
    def need_update(self):
        # type: () -> bool

        # Ни разу не обновляли
        if self.update_started is None:
            return True

        # Обновление идет и еще не слишком долго
        if (
            (self.updated is None or self.updated < self.update_started)
            and self.update_started > datetime.utcnow() - timedelta(seconds=settings.MAX_TARIFFS_UPDATE_TIME)
        ):
            return False

        # Данных нет или они устарели
        if self.status != TariffKeyDataStatus.ACTUAL:
            return True

        # Был получен пустой список тарифов и уже прошло достаточное время
        if not self.tariffs and datetime.utcnow() - self.updated > timedelta(seconds=self.empty_tariffs_ttl):
            return True

        return False


class FromToKeyCacheAsynchTariffsProvider(TariffsProvider):
    """
    Провайдер тарифов электричек, получающий и кэширующий данные по ключам (станция от, станция до, дата)
    Если актуальных тарифов нет в кэше, то получение их из внешнего API делается асинхронно
    """

    @traced_function
    def get_tariffs(self, tariff_keys):
        # type: (Iterable[TariffKey]) -> Dict[TariffKey, TariffKeyData]
        """Получение тарифов по ключам тарифов"""

        if not self.check_get_tariffs_enabled():
            return {}

        from_to_keys = {get_tariff_from_to_key(tariff_key) for tariff_key in tariff_keys}

        tariffs_by_from_to_keys = self.get_selling_tariffs_by_from_to_keys(from_to_keys)

        tariffs_by_tariff_keys = {}
        for tariff_key in tariff_keys:
            from_to_key = get_tariff_from_to_key(tariff_key)
            if from_to_key in tariffs_by_from_to_keys:
                tariffs_by_tariff_keys[tariff_key] = TariffKeyData([], tariffs_by_from_to_keys[from_to_key].status)
                for tariff in tariffs_by_from_to_keys[from_to_key].tariffs:
                    if (
                        tariff.tariff_type == tariff_key.tariff_type
                        and tariff.partner in SUBURBAN_CARRIERS_BY_COMPANIES_ID[tariff_key.company]
                    ):
                        tariffs_by_tariff_keys[tariff_key].tariffs.append(tariff)

        return tariffs_by_tariff_keys

    def get_selling_tariffs_by_from_to_keys(self, from_to_keys):
        # type: (Iterable[TariffFromToKey]) -> Dict[TariffFromToKey, TariffKeyData]
        """Получение тарифов по ключам from-to-date"""

        tariffs_by_from_to_keys = self.get_tariffs_data_from_cache(from_to_keys)

        keys_to_update = [
            key for key in from_to_keys
            if key not in tariffs_by_from_to_keys or tariffs_by_from_to_keys[key].need_update
        ]
        if keys_to_update:
            self.run_get_tariffs_data_from_provider(keys_to_update)

        return self.make_selling_tariffs_from_tariffs_data(tariffs_by_from_to_keys)

    def check_get_tariffs_enabled(self):
        # type: () -> bool
        raise NotImplementedError

    def get_tariffs_data_from_cache(self, from_to_keys):
        # type: (Iterable[TariffFromToKey]) -> Dict[TariffFromToKey, BaseFromToKeyTariffsData]
        """Получение тарифов из монги"""
        raise NotImplementedError

    def run_get_tariffs_data_from_provider(self, from_to_keys):
        # type: (List[TariffFromToKey]) -> None
        """Запуск асинхронного получения данных от провайдера, для тех ключей, для которых нужно"""
        raise NotImplementedError

    def make_selling_tariffs_from_tariffs_data(self, tariffs_data_for_keys):
        # type: (Dict[TariffFromToKey, Optional[BaseFromToKeyTariffsData]]) -> Dict[TariffFromToKey, TariffKeyData]
        """Формирование выходных данных по полученным тарифам"""
        raise NotImplementedError


class FromToKeyCacheTariffsProvider(TariffsProvider):
    """Провайдер тарифов электричек, получающий и кэширующий данные по ключам (станция от, станция до, дата)"""

    @traced_function
    def get_tariffs(self, tariff_keys):
        # type: (Iterable[TariffKey]) -> Dict[TariffKey, TariffKeyData]
        """Получение тарифов по ключам тарифов"""

        if not self.check_get_tariffs_enabled():
            return {}

        from_to_keys = {get_tariff_from_to_key(tariff_key) for tariff_key in tariff_keys}

        tariffs_by_from_to_keys = self.get_selling_tariffs_by_from_to_keys(from_to_keys)

        tariffs_by_tariff_keys = {}
        for tariff_key in tariff_keys:
            from_to_key = get_tariff_from_to_key(tariff_key)
            if from_to_key in tariffs_by_from_to_keys:
                tariffs_by_tariff_keys[tariff_key] = TariffKeyData([], tariffs_by_from_to_keys[from_to_key].status)
                for tariff in tariffs_by_from_to_keys[from_to_key].tariffs:
                    if (
                        tariff.tariff_type == tariff_key.tariff_type
                        and tariff.partner in SUBURBAN_CARRIERS_BY_COMPANIES_ID[tariff_key.company]
                    ):
                        tariffs_by_tariff_keys[tariff_key].tariffs.append(tariff)

        return tariffs_by_tariff_keys

    def get_selling_tariffs_by_from_to_keys(self, from_to_keys):
        # type: (Iterable[TariffFromToKey]) -> Dict[TariffFromToKey, TariffKeyData]
        """Получение тарифов по ключам from-to-date"""

        tariffs_by_from_to_keys = self.get_tariffs_data_from_cache(from_to_keys)

        keys_with_no_tariffs = set(key for key in from_to_keys if not tariffs_by_from_to_keys.get(key))
        for from_to_key in keys_with_no_tariffs:
            tariffs_data = self.get_tariffs_data_from_provider(from_to_key)
            if tariffs_data:
                tariffs_by_from_to_keys[from_to_key] = tariffs_data
                self.save_tariffs_data_to_cache(from_to_key, tariffs_data)

        return self.make_selling_tariffs_from_tariffs_data(tariffs_by_from_to_keys)

    def check_get_tariffs_enabled(self):
        # type: () -> bool
        raise NotImplementedError

    def get_tariffs_data_from_cache(self, from_to_keys):
        # type: (Iterable[TariffFromToKey]) -> Dict[TariffFromToKey, Any]
        """Получение тарифов из монги"""
        raise NotImplementedError

    def get_tariffs_data_from_provider(self, from_to_key):
        # type: (TariffFromToKey) -> Any
        """Получение тарифов от провайдера"""
        raise NotImplementedError

    def save_tariffs_data_to_cache(self, from_to_key, tariffs_data_for_key):
        # type: (TariffFromToKey, Any) -> ()
        """Запись тарифов в монгу"""
        raise NotImplementedError

    def make_selling_tariffs_from_tariffs_data(self, tariffs_data_for_all_keys):
        # type: (Dict[TariffFromToKey, Any]) -> Dict[TariffFromToKey, TariffKeyData]
        """Формирование выходных данных по полученным тарифам"""
        raise NotImplementedError
