# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
from typing import List, Dict, Optional, AnyStr, Iterable, Set
from datetime import datetime
from attr import attrs, attrib

from common.utils.namedtuple import namedtuple_with_defaults

from travel.rasp.suburban_selling.selling.tariffs.tariffs_configuration import TariffsConfiguration
from travel.rasp.suburban_selling.selling.tariffs.selling_companies import SuburbanCarrierCompany


# Ключ для получения тарифов
TariffKey = namedtuple_with_defaults(
    'TariffKey', [
        'date',          # type: datetime.date
        'station_from',  # type: int
        'station_to',    # type: int
        'company',       # type: int  # rasp company id
        'tariff_type',   # type: str  # rasp tariff type code
    ],
    defaults={'tariff_type': None}
)


class SellingTariff(object):
    def __init__(
        self, tariff_type, partner, name, description, price,
        max_days, valid_from, valid_until, book_data, selling_flow
    ):
        # type: (AnyStr, AnyStr, AnyStr, AnyStr, float, Optional[int], datetime, datetime, Dict, AnyStr) -> None

        self.tariff_id = None
        self.partner = partner
        self.tariff_type = tariff_type
        self.name = name
        self.description = description
        self.price = price
        self.max_days = max_days
        self.valid_from = valid_from
        self.valid_until = valid_until
        self.book_data = book_data
        self.selling_flow = selling_flow

    def set_tariff_id(self, tariff_id):
        # type: (int) -> None
        self.tariff_id = tariff_id

    def to_json(self):
        # type: () -> Dict
        return {
            'id': self.tariff_id,
            'partner': self.partner,
            'type': self.tariff_type,
            'name': self.name,
            'description': self.description,
            'price': self.price,
            'max_days': self.max_days,
            'valid_from': self.valid_from.isoformat(),
            'valid_until': self.valid_until.isoformat(),
            'book_data': json.dumps(self.book_data),
            'selling_flow': self.selling_flow
        }


def carrier_to_json(carrier_company):
    # type: (SuburbanCarrierCompany) -> Dict
    return {
        'code': carrier_company.code,
        'provider': carrier_company.provider,
        'title': carrier_company.title,
        'ogrn': carrier_company.ogrn,
        'address': carrier_company.address,
        'work_time': carrier_company.work_time
    }


class TariffKeyDataStatus(object):
    """Статус получения тарифов от провайдера по ключу"""
    NO_DATA = 0  # Данные не получены
    OLD = 1  # Получены устаревшие данные
    ACTUAL = 2  # Получены актуальные данные


@attrs
class TariffKeyData(object):
    tariffs = attrib(type=List[SellingTariff])
    status = attrib(type=TariffKeyDataStatus)


class TariffsProvider(object):
    """Провайдер для получения тарифов электричек"""

    def __init__(self, tariffs_configuration):
        # type: (TariffsConfiguration) -> None
        self.tariffs_configuration = tariffs_configuration

    def set_tariffs_getter(self, tariffs_getter):
        self.tariffs_getter = tariffs_getter

    def get_selling_flows(self):
        # type: () -> Set[AnyStr]
        """Поддерживаемые флоу продаж"""
        raise NotImplementedError

    def get_tariffs(self, tariff_keys):
        # type: (Iterable[TariffKey]) -> Dict[TariffKey, TariffKeyData]
        """
        :param tariff_keys: множество ключей тарифов
        :return Словарь списков тарифов, разложенных по ключам
        """
        raise NotImplementedError
