# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from attr import attrs, attrib
from typing import AnyStr, Iterable, Optional

from common.models.schedule import Company
from common.models.tariffs import SuburbanSellingBarcodePreset


@attrs
class SuburbanTariffType(object):
    code = attrib(type=AnyStr)
    tariff_type_id = attrib(type=int)
    description = attrib(default=None, type=Optional[AnyStr])


class SuburbanCarrierCode(object):
    CPPK = 'cppk'
    AEROEXPRESS = 'aeroexpress'
    SZPPK = 'szppk'
    MTPPK = 'mtppk'
    BASHPPK = 'bashppk'
    SODRUZHESTVO = 'sodruzhestvo'


@attrs
class SuburbanCarrierCompany(object):
    code = attrib(type=AnyStr)
    provider = attrib(type=AnyStr)
    title = attrib(type=AnyStr)
    selling_enabled_prod = attrib(type=bool, default=False)
    selling_enabled_testing = attrib(type=bool, default=False)
    tariff_types = attrib(default=[], type=Optional[Iterable[SuburbanTariffType]])
    ogrn = attrib(default=None, type=Optional[AnyStr])
    address = attrib(default=None, type=Optional[AnyStr])
    work_time = attrib(default=None, type=Optional[AnyStr])
    code_in_provider = attrib(default=None, type=Optional[AnyStr])
    barcode_preset = attrib(default=None, type=Optional[AnyStr])


SUBURBAN_CARRIERS_BY_CODES = {
    'aeroexpress': SuburbanCarrierCompany(
        code='aeroexpress',
        provider='aeroexpress',
        title='ООО «Аэроэкспресс»',
        selling_enabled_prod=True,
        selling_enabled_testing=True,
        ogrn='1055009320695',
        address='141425, Московская обл., г. Химки, аэропорт «Шереметьево-2»б а/я 170',
        work_time='Режим работы офисов с 8:45 до 17:15. Часы работы терминалов с 5:00 утра до 1:00 ночи',
        tariff_types=[
            SuburbanTariffType(
                code='aeroexpress',
                tariff_type_id=58,
            )
        ]
    ),
    SuburbanCarrierCode.CPPK: SuburbanCarrierCompany(
        code=SuburbanCarrierCode.CPPK,
        provider='movista',
        title='АО «Центральная ППК» (АО «ЦППК»)',
        selling_enabled_prod=True,
        selling_enabled_testing=True,
        ogrn='1057749440781',
        address='115054, Москва, Павелецкая пл., 1А',
        work_time=None,
        barcode_preset=SuburbanSellingBarcodePreset.PDF417_CPPK,
        tariff_types=[
            SuburbanTariffType(
                code='etrain',
                tariff_type_id=1,
                description='Билет подходит для электричек, «Стандарт плюс», «Иволга» и Аэроэкспрессов на участке Одинцово – Окружная'
            ),
            SuburbanTariffType(
                code='express',
                tariff_type_id=2,
                description='Билет подходит для поездов «РЭКС», «Спутник», «Скорый», а также обычных электричек и «гибридных» фирменных экспрессов'
            )
        ]
    ),
    SuburbanCarrierCode.SZPPK: SuburbanCarrierCompany(
        code=SuburbanCarrierCode.SZPPK,
        code_in_provider='СЗППК',
        provider='im',
        title='АО «Северо-Западная ППК» (АО «СЗППК»)',
        selling_enabled_prod=True,
        selling_enabled_testing=True,
        ogrn='1067847492888',
        address='190068, г. Санкт-Петербург, проспект Римского-Корсакова, д.47 лит.А',
        work_time=None,
        barcode_preset=SuburbanSellingBarcodePreset.PDF417_SZPPK,
        tariff_types=[
            SuburbanTariffType(
                code='etrain',
                tariff_type_id=1,
                description='Билет на электричку стандарт (обычный пригородный поезд)'
            ),
            SuburbanTariffType(
                code='last',
                tariff_type_id=85,
                description='Билет на электричку «Ласточка»'
            ),
            SuburbanTariffType(
                code='komfort',
                tariff_type_id=229,
                description='Билет на электричку «Комфорт»'
            )
        ]
    ),
    SuburbanCarrierCode.MTPPK: SuburbanCarrierCompany(
        code=SuburbanCarrierCode.MTPPK,
        code_in_provider='МТППК',
        provider='im',
        title='АО «Московско-Тверская ППК» (АО «МТППК»)',
        selling_enabled_prod=True,
        selling_enabled_testing=True,
        ogrn='1096952014390',
        address='107078, г. Москва, ул. Новорязанская 18, стр 21',
        work_time=None,
        barcode_preset=SuburbanSellingBarcodePreset.AZTEC_MTPPK,
        tariff_types=[
            SuburbanTariffType(
                code='etrain',
                tariff_type_id=1,
                description='Билет на электричку стандарт (обычный пригородный поезд)',
            ),
            SuburbanTariffType(
                code='last',
                tariff_type_id=85,
                description='Билет на электричку «Ласточка»'
            ),
            SuburbanTariffType(
                code='komfort',
                tariff_type_id=229,
                description='Билет на электричку «Комфорт»'
            )
        ]
    ),
    SuburbanCarrierCode.BASHPPK: SuburbanCarrierCompany(
        code=SuburbanCarrierCode.BASHPPK,
        code_in_provider='БАШППК',
        provider='im',
        title='АО «Башкортостанская ППК»',
        selling_enabled_prod=True,
        selling_enabled_testing=True,
        ogrn='1100280014020',
        address='450095, г. Уфа, Центральная, 13',
        work_time=None,
        barcode_preset=SuburbanSellingBarcodePreset.PDF417_SZPPK,
        tariff_types=[
            SuburbanTariffType(
                code='etrain',
                tariff_type_id=1,
                description='Билет на электричку стандарт (обычный пригородный поезд)',
            ),
            SuburbanTariffType(
                code='last',
                tariff_type_id=85,
                description='Билет на электричку «Ласточка»'
            ),
        ]
    ),
    SuburbanCarrierCode.SODRUZHESTVO: SuburbanCarrierCompany(
        code=SuburbanCarrierCode.SODRUZHESTVO,
        code_in_provider='СОДРУЖ',
        provider='im',
        title='АО «Содружество»',
        selling_enabled_prod=True,
        selling_enabled_testing=True,
        ogrn='1100280014020',
        address='420021, г. Казань, ул. Галиаскара Камала, 11',
        work_time=None,
        barcode_preset=SuburbanSellingBarcodePreset.PDF417_SZPPK,
        tariff_types=[
            SuburbanTariffType(
                code='etrain',
                tariff_type_id=1,
                description='Билет на электричку стандарт (обычный пригородный поезд)',
            ),
            SuburbanTariffType(
                code='last',
                tariff_type_id=85,
                description='Билет на электричку «Ласточка»'
            ),
        ]
    ),
}


SUBURBAN_CARRIERS_BY_COMPANIES_ID = {
    Company.CPPK_ID: [SuburbanCarrierCode.CPPK],
    Company.CPPK_AEROEX_ID: [SuburbanCarrierCode.CPPK, SuburbanCarrierCode.AEROEXPRESS],
    Company.AEROEXPRESS_ID: [SuburbanCarrierCode.AEROEXPRESS],
    Company.SZPPK_ID: [SuburbanCarrierCode.SZPPK],
    Company.MTPPK_ID: [SuburbanCarrierCode.MTPPK],
    Company.SZPPK_MTPPK_ID: [SuburbanCarrierCode.SZPPK, SuburbanCarrierCode.MTPPK],
    Company.BASHPPK_ID: [SuburbanCarrierCode.BASHPPK],
    Company.SAMPPK_BASHPPK_ID: [SuburbanCarrierCode.BASHPPK],
    Company.SODRUZHESTVO_ID: [SuburbanCarrierCode.SODRUZHESTVO]
}
