# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from typing import List, Dict, AnyStr
from collections import defaultdict

from travel.rasp.suburban_selling.selling.tariffs.selling_companies import SuburbanCarrierCompany, SuburbanTariffType


log = logging.getLogger(__name__)


class TariffsConfiguration(object):
    """Информация о перевозчиках и тарифах, которая берется из базы Расписаний"""

    def __init__(self, carriers_by_codes, carrier_codes_by_company_id):
        # type: (Dict[AnyStr, SuburbanCarrierCompany], Dict[int, List[AnyStr]]) -> None

        self.carriers_by_codes = carriers_by_codes                   # type: Dict[AnyStr, SuburbanCarrierCompany]
        self.tariff_types_by_carriers_and_codes = defaultdict(dict)  # type: Dict[AnyStr, Dict[AnyStr, SuburbanTariffType]]
        self.tariff_types_by_carriers_and_ids = defaultdict(dict)    # type: Dict[AnyStr, Dict[int, SuburbanTariffType]]
        for carrier in carriers_by_codes.values():
            self.tariff_types_by_carriers_and_codes[carrier.code] = {
                tariff_type.code: tariff_type for tariff_type in carrier.tariff_types
            }
            self.tariff_types_by_carriers_and_ids[carrier.code] = {
                tariff_type.tariff_type_id: tariff_type for tariff_type in carrier.tariff_types
            }

        self.carrier_codes_by_company_id = carrier_codes_by_company_id  # type: Dict[int, List[AnyStr]]
        self.provider_codes_by_companies_id = {}                        # type: Dict[int, List[AnyStr]]
        for company_id, carrier_codes in carrier_codes_by_company_id.items():
            self.provider_codes_by_companies_id[company_id] = [
                carriers_by_codes[code].provider for code in carrier_codes if code in carriers_by_codes
            ]

    def get_company_provider_codes(self, company_id):
        # type: (int) -> List[AnyStr]
        """По id перевозчика определяем, тарифы каких провайдеров имеет смысл запрашивать"""
        try:
            return self.provider_codes_by_companies_id[company_id]
        except KeyError:
            log.warning('Selling is not available for company %s', company_id)
            return []
