# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import json

from django.http import HttpResponse
from django.views.decorators.http import require_http_methods
from marshmallow import Schema, fields

from common.models.tariffs import SuburbanSellingFlow, SuburbanSellingBarcodePreset
from travel.rasp.suburban_selling.selling.tariffs.api import get_tariffs_for_tariff_keys, TariffKey


log = logging.getLogger(__name__)


class TariffKeySchema(Schema):
    date = fields.Date(default=None)
    station_from = fields.Int(required=True)
    station_to = fields.Int(required=True)
    tariff_type = fields.String(default=None)
    company = fields.Int(default=None)


class GetTariffsRequestSchema(Schema):
    keys = fields.Nested(TariffKeySchema, required=True, many=True)
    selling_flows = fields.List(fields.String, default=None)
    barcode_presets = fields.List(fields.String, default=None)
    tariff_types = fields.Dict(default=None)


class ProviderTariffsSchema(Schema):
    provider = fields.String(required=True)
    tariffs = fields.List(fields.Dict, many=True)


class KeyResponseSchema(Schema):
    key = fields.Nested(TariffKeySchema)
    tariff_ids = fields.List(fields.Int)


class SellingPartnerSchema(Schema):
    code = fields.String()
    provider = fields.String()
    title = fields.String()
    ogrn = fields.String()
    address = fields.String()
    work_time = fields.String()


class GetTariffsResponseSchema(Schema):
    selling_tariffs = fields.Nested(ProviderTariffsSchema, many=True)
    keys = fields.Nested(KeyResponseSchema, many=True)
    selling_partners = fields.Nested(SellingPartnerSchema, many=True)


@require_http_methods(['POST'])
def get_tariffs_view(request):
    """
    Запрос :
    {
        # Cписок ключей тарифов
        "keys": [
            {
                "date": "2020-10-12",
                "station_from": 42,  # rasp id
                "station_to": 43,  # rasp Id
                "company": 153,  # rasp company id
                "tariff_type": "express",  # rasp tariff type
                ... Здесь с появлением новых провайдеров могут добавляться новые поля (train_number и т.п.)
            },
            ...
        ],
        # Поддерживаемые типы флоу продаж
        "selling_flows": ["simple", "validator", "aeroexpress"],
        # Поддерживаемые пресеты настроек для формирования штрихкода
        "barcode_presets": ["PDF417_cppk", "PDF417_szppk", "Aztec_mtppk"],
        # Типы тарифов по номерам поездов (для ИМ)
        "tariff_types":{"6701": 1, "6703": 1, "7921": 85}
    }

    Ответ:
    {
        "result": {
            "selling_tariffs": [
                {
                    "provider": "movista",
                    "tariffs": [
                        # dict любого формата, описывающего данные тарифа. Зависит от провайдера
                        {
                            "id": 1,  # Порядковый номер в списке всех тарифов
                            "price": 56.12,
                            "name": "etrain",
                            "description": "Билет подходит для электричек, «Стандарт плюс», «Иволга», «Комфорт»",
                            "valid_from": "2020-10-12T00:00:00+03:00",
                            "valid_until": "2020-10-13T03:00:00+03:00",

                            # тут json в строке
                            "book_data": "{
                                "date": "2020-10-12",
                                "station_from": 2001305,  # express_id for movista
                                "station_to": 2000005,  # express_id for movista
                                "fare_id": 892304,
                            }"
                        },
                        ...
                    ]
                },
                ...
            ]
            "keys": [
                {
                    # ключ из запроса
                    "key": {
                        "date": "2020-10-12",
                        "station_from": 42,
                        "station_to": 43,
                        "company": 153,
                        "tariff_type": "express",
                    },
                    "tariff_ids": [1, 2]  # Список id тарифов из selling_tariffs для данного ключа
                },
                ...
            ],
            # Список партнеров-перевозчиков
            "selling_partners": [
                {
                    "code": "cppk",
                    "provider": "movista",
                    "title": "ППК",
                    "ogrn": "111",
                    "address": "Васюки",
                    "work_time": "До обеда"
                },
                ...
            ]
        },
        "errors": []
    }
    """
    tariffs_request, errors = GetTariffsRequestSchema().load(json.loads(request.body))

    tariff_keys = [TariffKey(**key) for key in tariffs_request['keys']]

    selling_flows = (
        tariffs_request.get('selling_flows')
        or [SuburbanSellingFlow.AEROEXPRESS, SuburbanSellingFlow.VALIDATOR]
    )
    barcode_presets = (
        tariffs_request.get('barcode_presets')
        or [SuburbanSellingBarcodePreset.PDF417_CPPK, SuburbanSellingBarcodePreset.PDF417_SZPPK]
    )

    tariff_type_ids_by_numbers = tariffs_request.get('tariff_types') or {}
    result = get_tariffs_for_tariff_keys(tariff_keys, selling_flows, barcode_presets, tariff_type_ids_by_numbers)

    data, errors = GetTariffsResponseSchema().dump(result)

    response = json.dumps({'result': data, 'errors': errors}, ensure_ascii=False)
    return HttpResponse(response, content_type='application/json; charset=utf-8')
