# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from xmlrpclib import ServerProxy, _datetime_type
from django.conf import settings

from common.data_api.tvm.instance import tvm_factory
from common.data_api.tvm.transport import TvmSafeTransport
from common.celery.task import single_launch_task
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date.environment import now
from common.utils.metrics import report_progress
from common.utils.try_hard import try_hard

from travel.rasp.suburban_selling.selling.aeroexpress.models import ClientContracts, ClientContract, AeroBilling


log = logging.getLogger(__name__)


define_setting('BALANCE_URL', env={
    Configuration.PRODUCTION: 'https://balance-xmlrpc-tvm.paysys.yandex.net:8004/xmlrpctvm',
}, default='https://balance-xmlrpc-tvm-ts.paysys.yandex.net:8004/xmlrpctvm')


def get_client_contracts(client_id):
    transport = TvmSafeTransport(tvm_factory.get_provider(), settings.TVM_BALANCE)
    server = ServerProxy(settings.BALANCE_URL, transport)
    return server.Balance.GetClientContracts({'ClientID': client_id})


@try_hard(max_retries=10, sleep_duration=5)
def update_contracts():
    now_dt = now()
    try:
        client_contracts = get_client_contracts(AeroBilling.partner_id)
        if not client_contracts:
            raise Exception('No contracts for Aeroex in Balance')

        def dt_from_xml(dt):
            return _datetime_type(dt.value) if dt else None

        contracts = []
        for client_contract in client_contracts:
            if AeroBilling.contract_service_id in client_contract['SERVICES']:
                contracts.append(ClientContract(
                    services=client_contract['SERVICES'],
                    start_dt=dt_from_xml(client_contract['DT']),
                    finish_dt=dt_from_xml(client_contract.get('FINISH_DT')),
                    is_active=client_contract['IS_ACTIVE'],
                    is_cancelled=client_contract['IS_CANCELLED'],
                    is_suspended=client_contract['IS_SUSPENDED'],
                    partner_commission_sum=client_contract['PARTNER_COMMISSION_SUM']
                ))
        ClientContracts.objects.create(updated_at=now_dt, contracts=contracts, partner_id=AeroBilling.partner_id)
        log.info('Успешно обновили контракты баланса')
        ClientContracts.objects(updated_at__lt=now_dt, partner_id=AeroBilling.partner_id).delete()
    except Exception as ex:
        log.error('Не смогли обновить контракты баланса: {}'.format(repr(ex)))
        raise


@single_launch_task()
@report_progress('client_contracts_update')
def update():
    update_contracts()
