# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from common.celery.task import single_launch_task
from common.data_api.movista.instance import movista_client
from common.models.geo import StationCode, CodeSystem
from common.db.mongo.bulk_buffer import BulkBuffer
from common.utils.metrics import report_progress

from travel.rasp.suburban_selling.selling.movista.models import MovistaStations


log = logging.getLogger(__name__)


@single_launch_task()
@report_progress('movista_load_stops_data')
def load_stops_data():
    try:
        stops = movista_client.stops()

        has_wicket_by_express_ids = {}
        wicket_type_by_express_ids = {}
        for stop in stops:
            try:
                has_wicket_by_express_ids[str(stop['id'])] = stop['hasWicket']
                wicket_type_by_express_ids[str(stop['id'])] = stop['tutorialType'] if stop['hasWicket'] else None
            except Exception:
                log.exception('Error in Movista stop. id={}'.format(stop.get('id', '')))

        codes = StationCode.objects.filter(system_id=CodeSystem.EXPRESS_ID, code__in=wicket_type_by_express_ids.keys())
        stations = []
        for station_code in codes:
            # Временный костыль до станции Реутов, пока в stops не будет приходить корректная информация
            if station_code.station_id == 9600796:
                station_dict = {
                    'station_id': 9600796,
                    'has_wicket': True,
                    'wicket_type': 'MID2Tutorial'
                }

            else:
                station_dict = {
                    'station_id': station_code.station_id,
                    'has_wicket': has_wicket_by_express_ids[station_code.code],
                    'wicket_type': wicket_type_by_express_ids[station_code.code]
                }
            stations.append(station_dict)

        with BulkBuffer(MovistaStations._get_collection(), max_buffer_size=5000, logger=log) as coll:
            for station_dict in stations:
                coll.update_one(
                    {'station_id': station_dict['station_id']},
                    {'$set': station_dict},
                    upsert=True
                )

    except Exception:
        log.exception('Movista stops data load error')
        raise
