# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import codecs
import csv
from copy import copy, deepcopy
from datetime import timedelta, datetime, time
from decimal import Decimal
from io import BytesIO

import openpyxl
from dateutil import parser
from django.conf import settings
from openpyxl.writer.excel import save_virtual_workbook
from pytils.dt import ru_strftime
from yql.api.v1.client import YqlClient

from common.celery.task import single_launch_task
from common.data_api.sendr.api import Campaign, Attachment
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.date import MSK_TZ
from travel.rasp.library.python.common23.date.environment import now_aware
from common.utils.metrics import report_progress


log = logging.getLogger(__name__)

define_setting(
    'YT_CPA_SUBURBAN_ORDERS_DIR',
    {Configuration.PRODUCTION: 'home/travel/prod/cpa/suburban/orders'},
    default='home/travel/testing/cpa/suburban/orders',
)

define_setting('SEND_ORDER_EMAILS', {
    Configuration.TESTING: ['martuginp@yandex-team.ru'],
    Configuration.PRODUCTION: [
        'martuginp@yandex-team.ru',
        'rasp-suburban-selling-report@yandex-team.ru',
        'shakti@yandex-team.ru'
    ],
}, default=[])

define_setting('AEROEX_SEND_ORDER_CAMPAIGN', {
    Configuration.PRODUCTION: 'W4RMUFX2-YK21',
}, default='8MCZYEX2-JT31')


AEROEX_FEE = Decimal('40.00')

EXCEL_MIME_TYPE = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
EXCEL_SUM_FORMAT = '=SUM({from_id}:{to_id})'


def _get_dt_bound(date):
    return (datetime.combine(date, time(0)) - timedelta(hours=3)).isoformat()


def get_yt_order_data(date_from, date_to, msk_now):
    log.info('Fill excel report. date_from={}, date_to={}'.format(date_from, date_to))
    data = []
    meta = {'total_sum': 0, 'fee': 0, 'count': 0, 'order_date': msk_now}

    query = """
        $get_dt_from_ts = ($ts) -> (
            CAST(DateTime::FromMilliseconds($ts * 1000) AS String)
        );
        SELECT provider_order_id, order_amount, departure_date, $get_dt_from_ts(created_at) as sell_dt
        FROM hahn.`{cpa_dir}`
        WHERE $get_dt_from_ts(created_at) >= '{date_from_bound}'
              and $get_dt_from_ts(created_at) < '{date_to_bound}'
              and provider = 'AEROEXPRESS'
              and provider_order_id is not null
              and status = 'confirmed'
    """.format(
        cpa_dir=settings.YT_CPA_SUBURBAN_ORDERS_DIR,
        date_from_bound=_get_dt_bound(date_from),
        date_to_bound=_get_dt_bound(date_to)
    )

    log.info('Run YQL query: {}'.format(query))

    with YqlClient(db='hahn', token=settings.YQL_TOKEN) as yql_client:
        request = yql_client.query(query)
        request.run()

        for table in request:
            for row in table.get_iterator():
                provider_order_id, order_amount, departure_date, sell_dt = row

                trip_date = (parser.parse(departure_date) + timedelta(days=1)).strftime('%d.%m.%Y')
                sell_date = (parser.parse(sell_dt) + timedelta(hours=3)).strftime('%d.%m.%Y')

                data.append({
                    'ticket_id': provider_order_id,
                    'sell_date': sell_date,
                    'sell_sum': order_amount,
                    'pay_date': sell_date,
                    'number': '',
                    'received': order_amount,
                    'trip_date': trip_date,
                    'name': 'физическое лицо'.encode('utf8')
                })
                meta['total_sum'] += order_amount
                meta['fee'] += AEROEX_FEE
                meta['count'] += 1

    return data, meta


def _comma_separated_value(value):
    return str(value).replace('.', ',')


def prepare_csv(data, meta):
    total_sum_str = _comma_separated_value(meta['total_sum'])
    csv_bytes = BytesIO()
    csv_bytes.write(codecs.BOM_UTF8)
    cw = csv.writer(csv_bytes, delimiter=str(';'))
    for entry in data:
        cw.writerow([entry['ticket_id'], entry['sell_date'], _comma_separated_value(entry['sell_sum']),
                     entry['pay_date'], entry['number'], _comma_separated_value(entry['received']),
                     entry['trip_date'], entry['name']])
    cw.writerow([total_sum_str, total_sum_str])
    cw.writerow([meta['count']])
    cw.writerow([total_sum_str])
    cw.writerow([_comma_separated_value(meta['fee'])])

    return csv_bytes.getvalue().strip(str('\r\n'))


def copy_excel_cell(new_sheet, i, j, cell, format_cell=None):
    new_cell = new_sheet.cell(row=i, column=j, value=cell.value)

    format_cell = format_cell or cell
    if format_cell.has_style:
        new_cell.font = copy(format_cell.font)
        new_cell.border = copy(format_cell.border)
        new_cell.fill = copy(format_cell.fill)
        new_cell.number_format = copy(format_cell.number_format)
        new_cell.alignment = copy(format_cell.alignment)


def merge_excel_cells(wb_sheet_1, entries_count):
    header_one_id = 7
    for column in range(2, 11):
        wb_sheet_1.merge_cells(
            start_row=header_one_id, start_column=column,
            end_row=header_one_id + 1, end_column=column
        )
    wb_sheet_1.merge_cells(
        start_row=header_one_id, start_column=11,
        end_row=header_one_id, end_column=13
    )

    header_two_id = 13 + entries_count
    wb_sheet_1.merge_cells(
        start_row=header_two_id, start_column=8,
        end_row=header_two_id, end_column=10
    )
    for column in range(2, 8):
        wb_sheet_1.merge_cells(
            start_row=header_two_id, start_column=column,
            end_row=header_two_id + 1, end_column=column
        )

    # merge text row
    for row in [11] + range(27, 31):
        wb_sheet_1.merge_cells(
            start_row=row + entries_count, start_column=2,
            end_row=row + entries_count, end_column=10
        )


def prepare_excel(data, meta):
    log.info('Fill excel file')
    wb_template = openpyxl.load_workbook('selling/templates/excel_order.xlsx')
    template_sheet_1 = wb_template.worksheets[0]

    wb = openpyxl.Workbook()

    # sheet_1 copy
    wb_sheet_1 = wb.active
    wb_sheet_1.title = template_sheet_1.title
    rows = list(template_sheet_1.rows)
    entries_start_id = 8
    data_row_template = rows[entries_start_id]

    # headers
    for i in range(0, entries_start_id):
        row = rows[i]
        for j, cell in enumerate(row, 1):
            copy_excel_cell(wb_sheet_1, i + 1, j, cell)

    # entries
    entries = []
    for entry in data:
        temp = deepcopy(data_row_template)
        temp[1].value = str(entry['ticket_id'])
        temp[2].value = entry['sell_date']
        temp[3].value = entry['sell_sum']
        temp[4].value = entry['pay_date']
        temp[6].value = entry['received']
        temp[7].value = entry['trip_date']
        entries.append(temp)

    entries_count = len(entries)
    total_row_id = entries_start_id + 1
    after_entries_id = total_row_id + entries_count

    for i, row in enumerate(entries, total_row_id):
        for j, cell in enumerate(row, 1):
            copy_excel_cell(wb_sheet_1, i, j, cell, format_cell=data_row_template[j - 1])

    # total
    total_row = rows[total_row_id]
    for j, cell in enumerate(total_row, 1):
        if j in {4, 7}:  # sum columns
            if entries:
                from_id, to_id = '{}{}'.format(cell.column_letter, total_row_id), '{}{}'.format(cell.column_letter,
                                                                                                after_entries_id - 1)
                cell.value = EXCEL_SUM_FORMAT.format(from_id=from_id, to_id=to_id)
            else:
                cell.value = 0
        copy_excel_cell(wb_sheet_1, after_entries_id, j, cell)

    # other
    for i in range(total_row_id + 1, len(rows)):
        row = rows[i]
        for j, cell in enumerate(row, 1):
            copy_excel_cell(wb_sheet_1, i + entries_count, j, cell)

    # merge cells
    merge_excel_cells(wb_sheet_1, entries_count)

    # fill meta
    wb_sheet_1.cell(20 + entries_count, 2).value = '=D{}'.format(after_entries_id)
    wb_sheet_1.cell(23 + entries_count, 2).value = meta['fee']
    wb_sheet_1.cell(17 + entries_count, 6).value = meta['count']

    date_to = meta['date_to'] - timedelta(days=1)
    period_text = 'за период с {} по {} года составляет'.format(
        ru_strftime('%d', meta['date_from'], inflected=True).lstrip('0'),
        ru_strftime('%d %B %Y', date_to, inflected=True)
    )

    wb_sheet_1.cell(17 + entries_count, 2).value = 'Общее количество билетов {}'.format(period_text)
    wb_sheet_1.cell(19 + entries_count,
                    2).value = '3. Общая сумма денежных средств за реализованные билеты (за вычетом возвратов) {}'.format(
        period_text)

    wb_sheet_1.cell(3, 3).value = ru_strftime('%d %B', meta['date_from'], inflected=True)
    wb_sheet_1.cell(3, 5).value = ru_strftime('%d %B', date_to, inflected=True)
    wb_sheet_1.cell(3, 6).value = '{} года'.format(date_to.strftime('%Y'))
    wb_sheet_1.cell(3, 13).value = meta['order_date'].strftime('%d.%m.%Y')

    # columns width
    for column_cells in wb_sheet_1.columns:
        length = max(len(cell.value or '') for cell in column_cells[6:8])
        wb_sheet_1.column_dimensions[column_cells[0].column_letter].width = length * 1.5

    for letter in ['C', 'D', 'E', 'K', 'L', 'M']:
        wb_sheet_1.column_dimensions[letter].width = 20

    # sheet_2 copy
    wb_sheet_2 = wb.create_sheet(wb_template.worksheets[1].title)
    for i, row in enumerate(wb_template.worksheets[1].rows, 1):
        for j, cell in enumerate(row, 1):
            copy_excel_cell(wb_sheet_2, i, j, cell)

    wb_sheet_2.merged_cells = wb_template.worksheets[1].merged_cells
    wb_sheet_2.row_dimensions[8].height = 25
    for column_cells in wb_sheet_2.columns:
        wb_sheet_2.column_dimensions[column_cells[0].column_letter].width = 12

    excel_data = save_virtual_workbook(wb)
    log.info('Excel report is filled successfully')
    return excel_data


def _get_range(msk_now):
    if 1 <= msk_now.day <= 10:
        previous_month = msk_now.replace(day=1) - timedelta(days=1)
        date_from = datetime(previous_month.year, previous_month.month, day=21)
        date_to = datetime(msk_now.year, msk_now.month, day=1)
    elif 11 <= msk_now.day <= 20:
        date_from = datetime(msk_now.year, msk_now.month, 1)
        date_to = date_from.replace(day=11)
    elif 21 <= msk_now.day:
        date_from = datetime(msk_now.year, msk_now.month, 11)
        date_to = date_from.replace(day=21)
    else:
        raise Exception('Невалидная дата')

    return MSK_TZ.localize(date_from), MSK_TZ.localize(date_to)


def prepare_excel_order(fake_now=None):
    msk_now = fake_now or now_aware()
    date_from, date_to = _get_range(msk_now)
    data, meta = get_yt_order_data(date_from=date_from, date_to=date_to, msk_now=msk_now)
    meta.update({'date_from': date_from, 'date_to': date_to})
    excel_data = prepare_excel(data, meta)

    return excel_data, meta


def prepare_excel_order_from_to(date_from, date_to):
    data, meta = get_yt_order_data(date_from=date_from, date_to=date_to, msk_now=now_aware())
    meta.update({'date_from': date_from, 'date_to': date_to})
    excel_data = prepare_excel(data, meta)

    return excel_data, meta


def prepare_csv_order(fake_now=None):
    msk_now = fake_now or now_aware()
    date_from, date_to = _get_range(msk_now)
    data, meta = get_yt_order_data(date_from=date_from, date_to=date_to, msk_now=msk_now)
    csv_data = prepare_csv(data, meta)

    return csv_data, meta


def prepare_csv_order_from_to(date_from, date_to):
    data, meta = get_yt_order_data(date_from=date_from, date_to=date_to, msk_now=now_aware())
    csv_data = prepare_csv(data, meta)

    return csv_data, meta


def send_order_emails(emails, data, meta):
    day = meta['date_from'].day
    if day < 11:
        decade = 1
    elif day < 21:
        decade = 2
    else:
        decade = 3

    filename = 'Яндекс_Извещение_Агента_за_{}_декаду_{}.xlsx'.format(
        decade, ru_strftime('%B_%Y', meta['date_from'], inflected=True)
    )
    campaign_send = Campaign.create_rasp_campaign(settings.AEROEX_SEND_ORDER_CAMPAIGN)
    attachment = Attachment(
        filename=filename,
        mime_type=EXCEL_MIME_TYPE,
        content=data
    )

    emails = emails if emails else settings.SEND_ORDER_EMAILS
    for email in emails:
        campaign_send.send(to_email=email, args={}, attachments=[attachment])


@single_launch_task()
@report_progress('send_aeroex_order_email')
def send_aeroex_order_email(emails=None, fake_now=None):
    try:
        excel_data, meta = prepare_excel_order(fake_now=fake_now)
        send_order_emails(emails, excel_data, meta)
    except Exception:
        raise
