# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from typing import Dict

from django.conf import settings
from django.http import JsonResponse
from django.views.decorators.http import require_http_methods

from common.data_api.tvm.instance import tvm_factory
from travel.rasp.suburban_selling.selling.models import Order


log = logging.getLogger(__name__)


@require_http_methods(['POST'])
def gdpr_takeout(request):
    """
    https://st.yandex-team.ru/RASPFRONT-6578
    Загрузка всех данных о всех билетах на Аэроэкспресс для указанного пользователя
    """
    if not _has_access(request):
        return JsonResponse({
            'status': 'error',
            'message': 'access forbidden',
        })

    passport_uid = request.POST['uid']
    unixtime = request.POST['unixtime']

    logging.info('Fetch data for %s in %s', passport_uid, unixtime)

    try:
        orders = _get_orders_by_passport_uid(passport_uid)
        if not orders:
            log.info('Can not find anything for %s in %s', passport_uid, unixtime)
            return JsonResponse({
                'status': 'no_data',
            })

        log.info('Can find something for %s in %s', passport_uid, unixtime)
        return JsonResponse({
            'status': 'ok',
            'data': {
                'orders.json': json.dumps(orders)
            }
        })

    except Exception:
        log.exception('ERROR /api/gdpr %s %s', passport_uid, unixtime)
        return JsonResponse({
            'status': 'error',
            'message': 'internal error',
        })


def _has_access(request):
    try:
        service_ticket = request.META.get('HTTP_X_YA_SERVICE_TICKET')
        st = tvm_factory.get_provider().check_service_ticket(service_ticket)
        allow = st.src in settings.TAKEOUT_ALLOWED_CONSUMERS
        if not allow:
            log.warn('The consumer with id [%s] is not allowed', st.src)
        return allow

    except Exception:
        log.exception('TvmException')
        return False


def _get_orders_by_passport_uid(passport_uid):
    query = {'user_info.passport_uid': int(passport_uid)}
    orders = Order.objects.filter(__raw__=query)
    return [_get_aeroexpress_order_info(order) for order in orders]


def _get_aeroexpress_order_info(order):
    # type: (Order) -> Dict

    order_dict = json.loads(order.to_json())

    passengers = []
    for p in order_dict['passengers']:
        data = {
            'first_name': p.get('first_name'),
            'surname': p.get('surname'),
            'patronymic_name': p.get('patronymic_name'),
        }
        ticket = p.get('ticket')
        if ticket:
            data['ticket'] = {
                'ticket_url': '{}&type=html'.format(ticket['ticket_url']),
                'code_url': '{}&type=qr'.format(ticket['ticket_url']),
                'price': ticket['ticket_price'],
                'tariff': ticket['tariff'],
                'route': ticket['st_depart'],
                'trip_date': ticket['trip_date']
            }

        passengers.append(data)

    return {
        'status': order['api_status'],
        'create_dt': order.created_at.isoformat() if order.created_at else None,
        'email': order['user_info']['email'],
        'phone': order['user_info']['phone'],
        'passengers': passengers
    }
