# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import json

from django.http import JsonResponse, HttpResponse
from django.views.decorators.http import require_http_methods

from common.data_api.travelers.client import DocumentTypes
from common.data_api.travelers.instance import travelers
from common.utils.blackbox_wrapper import Attributes, SessionInvalid
from travel.rasp.suburban_selling.selling.views.utils import get_blackbox_info


log = logging.getLogger(__name__)


def prepare_passport_gender(gender):
    if gender == 'f':
        return 'female'
    if gender == 'm':
        return 'male'


def _merge_with_passport_info(user_data, passport_info):
    ''' Данные из записной книжки обогащаем данными из паспорта.'''

    passport_emails = [email['address'] for email in passport_info.get('emails') or []]
    for email in passport_emails:
        if email not in user_data['emails']:
            user_data['emails'].append(email)

    passport_phones = [phone['attributes']['1'] for phone in passport_info.get('phones') or []]
    for phone in passport_phones:
        if phone not in user_data['phones']:
            user_data['phones'].append(phone)

    attrs = passport_info.get('attributes')
    if attrs:
        gender = attrs.get(str(Attributes.GENDER))
        if gender:
            user_data['gender'] = prepare_passport_gender(gender)

        birth_date = attrs.get(str(Attributes.BIRTHDAY))
        if birth_date:
            user_data['birth_date'] = birth_date

        first_name = attrs.get(str(Attributes.FIRST_NAME))
        if first_name:
            user_data['first_name'] = first_name

        last_name = attrs.get(str(Attributes.LAST_NAME))
        if last_name:
            user_data['last_name'] = last_name


@require_http_methods(['GET'])
def get_user_data(request):
    '''
    Получение данных о пользователе - из Записной книжки путешественника и Паспорта.
    https://st.yandex-team.ru/RASPFRONT-6373

    Для авторизации требуется OAuth-токен залогиненного юзера.
    '''
    try:
        passport_info = get_blackbox_info(
            request, getphones='bound', phone_attributes='1',
            attributes=[Attributes.BIRTHDAY, Attributes.GENDER, Attributes.FIRST_NAME, Attributes.LAST_NAME],
        )
    except SessionInvalid:
        return JsonResponse({'error': 'Unauthorized'}, status=403)
    except Exception:
        log.exception('Unable to get passport data')
        raise

    user_data = {'emails': [], 'phones': []}
    passengers = []
    user_uid, user_ticket = passport_info['uid'], passport_info['user_ticket']
    traveler = travelers.get_traveler(user_uid, user_ticket)
    if traveler:
        user_data['emails'].append(traveler['email'])
        user_data['phones'].extend(ph for ph in [traveler.get('phone'), traveler.get('phone_additional')] if ph)

        traveler_passengers = travelers.get_traveler_documents(user_uid, user_ticket, ['passengers', 'documents'])
        for passenger_data in traveler_passengers.get('passengers', []):
            docs = [d for d in passenger_data['documents'] if d['type'] == DocumentTypes.RU_NATIONAL_PASSPORT]

            if docs:
                passenger = {
                    'title': passenger_data['title'],
                    'birth_date': passenger_data['birth_date'],
                    'gender': passenger_data['gender'],
                    'documents': [],
                }
                passengers.append(passenger)

                for doc in docs:
                    suburban_doc = {
                        'number': doc['number'],
                        'type': doc['type'],
                        'first_name': doc['first_name'] or doc['first_name_en'],
                        'middle_name': doc['middle_name'] or doc['middle_name_en'],
                        'last_name':  doc['last_name'] or doc['last_name_en']
                    }
                    passenger['documents'].append(suburban_doc)

    _merge_with_passport_info(user_data, passport_info)

    return HttpResponse(json.dumps({
        'user': user_data,
        'passengers': passengers,
    }, ensure_ascii=False), content_type='application/json; charset=utf-8')
