# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os
import socket
import traceback

import pymongo
from django.db import connection
from django.conf import settings
from django.http import HttpResponse, JsonResponse
from django.views.decorators.http import require_http_methods

from common.db.mongo import databases
from common.db.mongo.counter import MongoCounter

from travel.rasp.suburban_selling.selling.views.utils import staff_login_required


def safe(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception:
            return traceback.format_exc()

    return wrapper


@safe
def get_hostname():
    return socket.gethostname()


@safe
def get_mysql_host():
    c = connection.cursor()
    c.execute('select @@hostname;')
    c.execute('show variables like "hostname";')
    return list(c.fetchall())[0][1]


@safe
def get_mongo_host(db_alias='default'):
    db = databases[db_alias]

    try:
        return [db.client.address]
    except pymongo.errors.InvalidOperation:
        return list(db.client.nodes)


@safe
def get_qloud_env():
    return {k: v for k, v in os.environ.items() if 'qloud' in k.lower()}


@require_http_methods(['GET'])
def ping(request):
    return HttpResponse('pong!', content_type='text/plain')


@staff_login_required
@require_http_methods(['GET'])
def env(request):
    return JsonResponse({
        'a': 1,
        'host': get_hostname(),
        'pid': os.getpid(),
        'qloud_env': get_qloud_env(),
        'mysql_host': get_mysql_host(),
        'mongo_hosts': get_mongo_host()
    })


@staff_login_required
@require_http_methods(['GET'])
def selling(request):
    db = databases['suburban_selling']
    counter = MongoCounter('visitors', db.counters)

    return JsonResponse({
        'visitors': counter.next_value(),
        'mongo_connected': get_mongo_host('suburban_selling'),
        'mongo_hosts': settings.MONGO_DATABASES['suburban_selling']['host']
    })
