# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import getpass
import os

import pymongo
from rasp_vault.api import get_secret

from common.settings import *  # noqa
from common.settings.configuration import BaseConfigurator, Configuration
from common.settings.utils import configure_raven, get_app_version
from travel.rasp.library.python.common23.logging.configure import configure_default_logging

PROJECT_PATH = os.path.abspath(os.path.join(os.path.dirname(__file__), ''))


ROOT_URLCONF = 'travel.rasp.suburban_selling.selling.urls'
LOG_PATH = '/var/log/app'

CELERY_WORKER_HIJACK_ROOT_LOGGER = False
CELERY_WORKER_PREFETCH_MULTIPLIER = 1
CELERY_TASK_ACKS_LATE = True

SQS_ENDPOINT = 'sqs.yandex.net:8771'
SQS_SECRET_KEY = 'not-used-yet'

OPENTRACING_TRACE_ALL = True
OPENTRACING_TRACED_ATTRIBUTES = ['path', 'method']

ALLOWED_HOSTS = ['*']


INSTALLED_APPS = COMMON_INSTALLED_APPS + COMMON_CONTRIB_APPS + [  # noqa
    'django_opentracing',
    'django.contrib.staticfiles',
    'django.contrib.admin',
    'django.contrib.humanize',
    'django.contrib.sessions',
    'django.contrib.sites',
    'raven.contrib.django.raven_compat',
    'travel.rasp.suburban_selling.selling',
]

MIDDLEWARE_CLASSES = [
    'django_opentracing.OpenTracingMiddleware',
    'travel.library.python.tracing.django.TracingMiddleware',
    'common.middleware.request_activator.RequestActivator',
    'common.middleware.syncdb.SyncDB',
    'common.middleware.now.Now',

    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.gzip.GZipMiddleware',
    'django.middleware.security.SecurityMiddleware',

    'common.middleware.yateamuser.YaTeamUserAuth'
]

# nginx (в т.ч. балансеры в awacs) ходит в приложение по http,
# при этом ставит HTTP_X_FORWARDED_PROTO, если изначальное соединение было по https.
# Эта настройка показывает Джанге, какой хедер читать, чтобы узнать, что соединение секьюрное.
SECURE_PROXY_SSL_HEADER = ('HTTP_X_FORWARDED_PROTO', 'https')

EMAIL_SENDER_DB_ALIAS = 'suburban_selling'

YASMAGENT_ITYPE = 'suburbanselling'
YASMAGENT_PROJECT = 'suburbanselling'

TRUST_PORT = 8028

AEROEX_CRT = '/etc/yandex/rasp/aero.crt'
AEROEX_KEY = '/etc/yandex/rasp/aero.key'

LOCK_DATABASE_NAME = 'suburban_selling'

TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [],
        'APP_DIRS': True,
        'OPTIONS': {
            'debug': False,
            'context_processors': [
                'django.contrib.auth.context_processors.auth',
            ]
        },
    },
]

PKG_VERSION = get_app_version()

PRECACHE_MANAGERS = [
    'common.models.schedule:Company.objects',
    'common.models.geo:Station.code_manager',
]


class SellingConfigurator(BaseConfigurator):
    def apply_base(self, settings):
        settings['TVM_DESTINATIONS'] = [
            'TVM_BLACKBOX', 'TVM_TRAVELERS', 'TVM_BLACKBOX_YATEAM', 'TVM_TRAVEL_API', 'TVM_BALANCE'
        ]

        settings['MONGO_DATABASES']['suburban_selling'] = {
            'port': 27018,
            'db': 'suburb',
            'user': 'suburb',
        }
        settings['TRUST_HOST'] = 'trust-payments-test.paysys.yandex.net'

        settings['MOS_RU_MDM_TOKEN'] = get_secret('suburban-tst-sudir3-mosru.mdm_test_token')

        base_log_level = os.environ.get('RASP_BASE_LOG_LEVEL', 'INFO')
        format_with_context = '%(levelname)s %(asctime)s %(context)s %(process)d %(name)s %(message)s'

        settings['LOGGING'] = configure_default_logging(settings, base_log_level, format_with_context,
                                                        jaeger_tracing=True)

        settings['LOGGING']['loggers'].update({
            'common.data_api': {
                'handlers': ['order_mongo_handler'],
                'level': 'INFO',
            },
            'common.email_sender': {
                'handlers': ['order_mongo_handler'],
                'level': 'INFO',
            },
            'selling.aeroexpress': {
                'handlers': ['order_mongo_handler'],
                'level': 'INFO',
            },
            'selling.views': {
                'handlers': ['order_mongo_handler'],
                'level': 'INFO',
            },
        })

        settings['LOGGING']['handlers'].update({
            'order_mongo_handler': {
                'class': 'travel.rasp.library.python.common23.logging.mongo.mongo_handler.MongoHandler',
                'dbalias': 'suburban_selling',
                'collection_name': 'order_logs',
                'filters': ['context_as_dict_filter', 'fields_in_context_filter'],
                'indexes': [pymongo.IndexModel('created_utc', background=True),
                            pymongo.IndexModel('context.order_uid', background=True)]
            }
        })

        settings['LOGGING']['filters'].update({
            'fields_in_context_filter': {
                '()': 'travel.rasp.library.python.common23.logging.FieldsInContextFilter',
                'required_fields': ['order_uid']
            },
        })

    def apply_testing(self, settings):
        settings['AEROEX_CRT_SECRET'] = 'rasp-suburban-selling-testing.AEROEX_CRT'
        settings['AEROEX_KEY_SECRET'] = 'rasp-suburban-selling-testing.AEROEX_KEY'

        settings['TVM_SERVICE_ID'] = 2002606

        settings['TRUST_SERVICE_TOKEN'] = get_secret('rasp-suburban-selling-testing.TRUST_SERVICE_TOKEN')
        settings['MONGO_DATABASES']['suburban_selling']['password'] = get_secret(
            'rasp-suburban-selling-testing.SUBURBAN_SELLING_MONGO_PASSWORD')
        selling_db = settings['MONGO_DATABASES']['suburban_selling']
        selling_db['dbaas_id'] = '4af117a2-5ff5-4da1-a5d1-268c79a5ac2b'
        selling_db['fallback_hosts'] = [  # на случай недоступности api dbaas
            'sas-uj88t6ugfx15pjsr.db.yandex.net',
            'iva-wiesbcng60ftjnf7.db.yandex.net',
            'man-g31kvrj0ip9hmp65.db.yandex.net',
        ]

        configure_raven(settings, 'suburban_selling')

        settings['BLACKBOX_URL'] = 'http://pass-test.yandex.ru/blackbox'

        settings['TAKEOUT_ALLOWED_CONSUMERS'] = {
            2001121,  # rasp-developers     https://abc.yandex-team.ru/services/rasp/resources/?show-resource=3260008
            2009783,  # takeout-testing  https://abc.yandex-team.ru/services/takeout/resources/?show-resource=4688915
            2010664,  # тестинг аудита безопасности
        }

    def apply_production(self, settings):
        settings['AEROEX_CRT_SECRET'] = 'rasp-suburban-selling-production.AEROEX_CRT'
        settings['AEROEX_KEY_SECRET'] = 'rasp-suburban-selling-production.AEROEX_KEY'

        settings['TVM_SERVICE_ID'] = 2002610

        settings['TRUST_SERVICE_TOKEN'] = get_secret('rasp-suburban-selling-production.TRUST_SERVICE_TOKEN')
        settings['MONGO_DATABASES']['suburban_selling']['password'] = get_secret(
            'rasp-suburban-selling-production.SUBURBAN_SELLING_MONGO_PASSWORD')
        selling_db = settings['MONGO_DATABASES']['suburban_selling']
        selling_db['dbaas_id'] = '6a133388-1a6d-4701-9ad7-4dc55d7609a0'
        selling_db['fallback_hosts'] = [  # на случай недоступности api dbaas
            'sas-j9ll6pnz1yv019lm.db.yandex.net',
            'vla-wcda76jn42iw8e2d.db.yandex.net',
            'man-wzphmzddwzvzdwkk.db.yandex.net',
        ]

        settings['TRUST_HOST'] = 'trust-payments.paysys.yandex.net'

        configure_raven(settings, 'suburban_selling')

        settings['TAKEOUT_ALLOWED_CONSUMERS'] = {
            2001121,  # rasp-developers     https://abc.yandex-team.ru/services/rasp/resources/?show-resource=3260008
            2009785,  # takeout-production  https://abc.yandex-team.ru/services/takeout/resources/?show-resource=4688916
        }


class SellingSQSConfigurator(BaseConfigurator):
    def apply(self, config_type, settings):
        super(SellingSQSConfigurator, self).apply(config_type, settings)

    def apply_testing(self, settings):
        settings['CELERY_TASK_DEFAULT_QUEUE'] = 'celery.default'
        settings['SQS_ACCESS_KEY'] = 'suburban-selling-test'

    def apply_production(self, settings):
        settings['CELERY_TASK_DEFAULT_QUEUE'] = 'celery.default'
        settings['SQS_ACCESS_KEY'] = 'suburban-selling'

    def apply_development(self, settings):
        settings['CELERY_TASK_DEFAULT_QUEUE'] = '{}.celery.default'.format(getpass.getuser())
        settings['SQS_ACCESS_KEY'] = 'suburban-selling-dev'

    def apply_after_all(self, settings):
        settings['CELERY_BROKER_TRANSPORT_OPTIONS'] = {
            'is_secure': False,
            'region': 'yandex'
        }
        self._setup_celery_broker_url(settings)

    @staticmethod
    def _setup_celery_broker_url(settings):
        settings['CELERY_BROKER_URL'] = 'sqs://{access_key}:{secret_key}@{endpoint}'.format(
            access_key=settings['SQS_ACCESS_KEY'],
            secret_key=settings['SQS_SECRET_KEY'],
            endpoint=settings['SQS_ENDPOINT']
        )


Configuration.configurator_classes += [SellingConfigurator, SellingSQSConfigurator]
