# coding: utf8
from django.utils.safestring import mark_safe
from django.db import models

from travel.rasp.suburban_tasks.suburban_tasks.utils import build_spec_buf_key

SIST_IST_HELP_TEXT = u'''
    Система–источник информации.
    Каждая информационная система, передающая информацию в централизованную базу данных, проставляет здесь код, однозначно идентифицирующий систему – источник информации:
    Первая цифра
    1 – ИСС ПС (в настоящее время не используется)
    2 – АРМ ТКНГДП
    3 – Изменения, внесенные администратором базы.
    Вторая и третья цифры- код дороги
'''
ID_XML_HELP_TEXT = u'''
    Идентификатор корректировочного сообщения.
    Целое число, однозначно идентифицирующее корректировочный документ.
'''
DATE_GVC_HELP_TEXT = u'''
    Дата и время загрузки корректировочного сообщения в ГВЦ.
'''
IDTR_HELP_TEXT = u'''
    Идентификатор поезда.
    Абстрактный числовой ключ поезда, к которому относится данное расписание в БД, уникальный для всех поездов в БД
'''


class LGDPPR_SCALENDAR(models.Model):
    """
    Уникальный идентификатор записи IDTR + CDATE
    """
    IDTR = models.IntegerField(null=False, help_text=IDTR_HELP_TEXT)
    IDR = models.IntegerField(null=False, help_text=u'''
        Идентификатор расписания, действующего для данного поезда в данную дату. =0, если поезд в данный день не ходит.
    ''')
    YEAR = models.IntegerField(null=False, help_text=u'''
        Целое число, означающее год ввода в действие графика движения поездов (ГДП), к которому относится данный поезд.
    ''')
    CDATE = models.DateField(null=False, help_text=u'''
        Дата, на которую задается расписание
    ''')
    OPER = models.IntegerField(null=False, help_text=u'''
        0 – первичное состояние (на момент ввода графика)
        1 – дополнительно назначен
        2 – отмена поезда в эту дату
        3 – изменение расписания в эту дату
    ''')

    class Meta:
        abstract = True
        verbose_name = u'Календарь|LGDPPR.SCALENDAR'
        verbose_name_plural = u'Календари|LGDPPR.SCALENDAR'
        index_together = [
            ('IDTR', 'CDATE')
        ]


class LGDPPR_SDOCS(models.Model):
    """
    Описание корректировочного документа.
    Таблица содержит реквизиты документов, являющихся основанием для корректировки расписания.
    P.S. пока у нас нет активного использования, и документация не очень хорошо представлена.
    """
    IDDOC = models.IntegerField(null=False)
    DOCTYPE = models.IntegerField(null=False)
    DOCNO = models.CharField(max_length=100, null=False)
    CDOR = models.IntegerField(null=False)
    DOCDATE = models.DateField(null=False)
    DOCINFO = models.CharField(max_length=500, null=True, default=None)

    class Meta:
        abstract = True


class LGDPPR_SRASPRP(models.Model):
    """
    Уникальный идентификатор записи IDTR + IDR + SEQ + IDRP
    """
    IDTR = models.IntegerField(null=False, help_text=u'''
        Абстрактный числовой ключ поезда, к которому относится данное расписание в БД, уникальный для всех поездов в БД
    ''')
    IDR = models.IntegerField(null=False, help_text=u'''
        Идентификатор расписания, действующего для данного поезда в данную дату. =0, если поезд в данный день не ходит.
    ''', db_index=True)
    CREG = models.IntegerField(null=False, help_text=u'''
        Числовой код региона движения пригородных поездов, к которому относится поезд.
    ''')
    SEQ = models.IntegerField(null=False, help_text=u'''
        Порядковый №  пункта в расписании
    ''')
    IDRP = models.IntegerField(null=False, help_text=u'''
        Уникальный идентификатор пункта расписания в ЦНСИ.
    ''')
    NUMP = models.IntegerField(null=False, help_text=u'''
        № поезда в пути следования по прибытии.
        Смена четности № поезда в пути следования зависит от установленного направления движения ж.д. линии
        № поезда в пути может изменяться относительно № по станции отправления по правилу:
        № в пути = № отправления+1, когда № отправления нечетный
        или
        № в пути = № отправления-1, когда № отправления четный.
    ''')
    NUMO = models.IntegerField(null=False, help_text=u'''
        № поезда в пути следования по отправлению, см NUMP
    ''')
    PRIB = models.TimeField(null=False, help_text=u'''
        Время прибытия
    ''')
    OTPR = models.TimeField(null=False, help_text=u'''
        Время отправления
    ''')
    KM = models.FloatField(null=False, help_text=u'''
        Километраж
        Расстояние в километрах от начала маршрута.
    ''')  # В документации рекомендуют тип DECIMAL (5,0)
    PRIBR = models.IntegerField(null=False, help_text=u'''
        Время в пути до прибытия.
        Время (в ½ минуты) движения от начального пункта до прибытия на  данный пункт
        0 – если время прибытия не определено.
    ''')
    OTPRR = models.IntegerField(null=False, help_text=u'''
        Время в пути до отправления.
        Время (в ½ минуты) движения от начального пункта до отправления с  данного пункта
        0 – если время отправления не определено.
    ''')
    TEXST = models.IntegerField(null=False, help_text=u'''
        Признак наличия посадки/высадки.
        0 – обычная стоянка
        1 – стоянка без посадки/высадки пассажиров
        3 – время прибытия/отправления и время в пути по станции не задано – в этом случае поля PRIB,
        OTPR содержат значение 00:01:00
    ''')

    class Meta:
        abstract = True
        verbose_name = u'Расписание|LGDPPR.SRASPRP'
        verbose_name_plural = u'Расписания|LGDPPR.SRASPRP'


class LGDPPR_STRAINS(models.Model):
    """
    Уникальный идентификатор записи IDTR
    """
    IDTR = models.IntegerField(null=False, help_text=ID_XML_HELP_TEXT, db_index=True)
    YEAR = models.IntegerField(null=False, help_text=u'''
        Год графика.
        Целое число, означающее год ввода в действие графика движения поездов (ГДП), к которому относится данный поезд.
        В БД будут храниться графики нескольких лет одновременно.
    ''')
    CREG = models.IntegerField(null=False, help_text=u'''
        Код пригородного региона.
        Числовой код региона движения пригородных поездов, к которому относится поезд.
    ''')
    NUM = models.IntegerField(null=False, help_text=u'''
        № поезда.
        Номер поезда в графике движения поездов.
    ''')
    LETTERS = models.CharField(max_length=100, null=True, default=None, help_text=u'''
        Буквенная часть номера.
        В пригородном движении к № поезда обычно добавляется буквенное сочетание, отражающее специфику поезда.
        Напр. «Р» - поезд курсирует только по Рабочим дням и т.п.
    ''')
    RP1 = models.IntegerField(null=False, help_text=u'''
        Идентификатор станции начала поезда.
        Уникальные идентификаторы станций начала и конца маршрута поезда в ЦНСИ.
    ''')
    RPK = models.IntegerField(null=False, help_text=u'''
        Идентификатор станции конца поезда. См. RP1
    ''')
    NAME = models.CharField(max_length=250, null=True, default=None, help_text=u'''
        Имя поезда.
        Символьная строка. Для поездов, имеющих индивидуально имя (напр. «Спутник», «Аэроэкспресс» и т.д.)
    ''')
    CATEGORY = models.IntegerField(null=False, help_text=u'''
        Категория поезда.
        1. Категория не задана;
        2. Обычный поезд;
        3. Ускоренный.
    ''')
    IDROOT = models.IntegerField(null=False, help_text=u'''
        Идентификатор маршрута.
        Уникальный идентификатор маршрута в таблице LGDPPR.SROOTS.
    ''')
    IDSOOB = models.IntegerField(null=False, help_text=u'''
        Идентификатор сообщения пригородных поездов.
        Уникальный идентификатор сообщения пригородного поезда  в таблице LGDPPR.SOOB.
    ''')
    CODEOWNER = models.IntegerField(null=False, help_text=u'''
        Код перевозчика.
        Целое число, соответствующее компании- перевозчику, в соответствии с классификатором перевозчиков.
    ''')
    NDAYS = models.CharField(max_length=100, null=True, default=None, help_text=u'''
        Периодичность обращения поезда.
        Текст, описывающий периодичность обращения поезда (напр. «по рабочим дням», «кроме пятницы»).
        Этот текст служит только для визуального отображения сведений о поезде.
    ''')
    STATE = models.IntegerField(null=False, help_text=u'''
        Статус поезда. Состояние данных о поезде:
        От 0 до 999999 –обычный поезд
        более 1000000 – составный поезд
    ''')
    IDR = models.IntegerField(null=False, help_text=u'''
        Идентификатор основного расписания.
        Идентификатор расписания , являющегося основным (нормативным) для данного поезда.
    ''')

    class Meta:
        abstract = True
        verbose_name = u'Поезд|LGDPPR.STRAINS'
        verbose_name_plural = u'Поезда|LGDPPR.STRAINS'


class LGDPPR_STRAINSVAR(models.Model):
    """
    Уникальный идентификатор записи IDR
    """
    IDTR = models.IntegerField(null=False, help_text=IDTR_HELP_TEXT)
    IDR = models.IntegerField(null=False, help_text=u'''
        Идентификатор расписания.
        Абстрактный числовой ключ расписания в БД.
    ''', db_index=True)
    YEAR = models.IntegerField(null=False, help_text=u'''
        Год графика.
        Целое число, означающее год ввода в действие графика движения поездов (ГДП),
        к которому относится данное расписание.
    ''')
    RP1 = models.IntegerField(null=False, help_text=u'''
        Идентификатор станции начала расписания.
        Уникальные коды станций начала и конца расписания в ЦНСИ.
    ''')
    RPK = models.IntegerField(null=False, help_text=u'''
       Идентификатор станции конца расписания. См. RP1
    ''')
    VPUTI = models.IntegerField(null=False, help_text=u'''
        Время в пути.
        Время в минутах движения от начального до конечного пункта расписания.
    ''')
    KM = models.FloatField(null=False, help_text=u'''
        Километраж
        Длина в километрах маршрута для данного расписания.
    ''')  # В документации рекомендуют тип DECIMAL (5,0)
    STOPS = models.IntegerField(null=False, help_text=u'''
        Время в стоянках.
        Суммарная длительность стоянок в данном расписании.
    ''')
    STATE = models.IntegerField(null=False, help_text=u'''
        Статус расписания.
        Состояние расписания:
        От 0 до 999999 – расписание обычного поезда
        более 1000000 – расписание составного поезда
    ''')
    CREG = models.IntegerField(null=False, help_text=u'''
        Код  пригородного  региона.
        Числовой код региона движения пригородных поездов, к которому относится поезд.
    ''')
    RASPTYPE = models.IntegerField(null=False, help_text=u'''
        Тип расписания.
        Число. Указывает тип данного расписания:
        0 – не определен;
        1 – основное расписание;
        2 – вариантное расписание;
        3 – исходное (состояние на начало действия графика) расписание
    ''')
    BASERASPID = models.IntegerField(null=False, help_text=u'''
        Родительское расписание.
        Идентификатор расписания, на основе которого создавалось данное расписание.
    ''')
    IDDOC = models.IntegerField(null=False, help_text=u'''
        Идентификатор документа.
        Идентификатор  документа в БД, на основании которого создавалось данное расписание.
        Предполагается, что реквизиты (и, возможно, содержание) документов на изменение расписания
        будут сохраняться в БД
    ''')

    class Meta:
        abstract = True
        verbose_name = u'Вариант расписания|LGDPPR.STRAINSVAR'
        verbose_name_plural = u'Варианты расписания|LGDPPR.STRAINSVAR'


class IC00_STAN(models.Model):
    STAN_ID = models.IntegerField(null=False)
    DOR_KOD = models.IntegerField(null=False)
    PRED_ID = models.IntegerField(null=True, default=None)
    OKATO_ID = models.IntegerField(null=False)
    ST_KOD = models.IntegerField(null=False)
    VNAME = models.CharField(max_length=250, null=False)
    NAME = models.CharField(max_length=250, null=False)
    STAN_TIP_ID = models.IntegerField(null=False)
    COR_TIP = models.CharField(max_length=100, null=False)
    DATE_ND = models.DateField(null=False)
    DATE_KD = models.DateField(null=False)
    COR_TIME = models.CharField(max_length=250, null=False)
    OPER_ID = models.IntegerField(null=True, default=None)
    REPL_FL = models.IntegerField(null=True, default=None)
    MNEM = models.CharField(max_length=250, null=True, default=None)

    class Meta:
        verbose_name = u'Станция|IC00.STAN'
        verbose_name_plural = u'Станции|IC00.STAN'


class Update(models.Model):
    FULL_UPDATE = 'full'
    CHANGES_UPDATE = 'changes'

    action_type = models.CharField(u'Тип выгрузки базы', max_length=255, choices=(
        (FULL_UPDATE, u'Выгрузка полного расписания'),
        (CHANGES_UPDATE, u'Выгрузка изменений'),
    ))

    updated_at = models.DateTimeField(auto_now_add=True, null=False, blank=False)
    last_gvc_date = models.DateTimeField(u'Последнее корректировка ГВЦ', null=False, blank=False,
                                         help_text=u'Дата последнего загруженного изменения')
    is_fake_gvc_date = models.BooleanField(u'ГВЦ Фейковое', default=True,
                                           help_text=u'Дата последнего изменения ГВЦ сформирована нами')

    query_from_dt = models.DateTimeField(u'Запрос от', null=True, blank=True)
    query_to_dt = models.DateTimeField(u'Запрос до', null=True, blank=True)

    class Meta:
        verbose_name = u'Выгрузка базы'
        verbose_name_plural = u'Выгрузки базы'

    def __unicode__(self):
        return u'Update {}: {}'.format(self.action_type, self.updated_at)


class Full_SCALENDAR(LGDPPR_SCALENDAR):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    class Meta(LGDPPR_SCALENDAR.Meta):
        verbose_name = u'Полное расписание. Календарь|LGDPPR.SCALENDAR'
        verbose_name_plural = u'Полное расписание. Календари|LGDPPR.SCALENDAR'


class Full_SDOCS(LGDPPR_SDOCS):
    update = models.ForeignKey('rzd_buffer.Update', null=False)


class Full_SRASPRP(LGDPPR_SRASPRP):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    class Meta(LGDPPR_SRASPRP.Meta):
        verbose_name = u'Полное расписание. Расписание|LGDPPR.SRASPRP'
        verbose_name_plural = u'Полное расписание. Расписания|LGDPPR.SRASPRP'


class Full_STRAINS(LGDPPR_STRAINS):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    class Meta(LGDPPR_STRAINS.Meta):
        verbose_name = u'Полное расписание. Поезд|LGDPPR.STRAINS'
        verbose_name_plural = u'Полное расписание. Поезда|LGDPPR.STRAINS'


class Full_STRAINSVAR(LGDPPR_STRAINSVAR):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    class Meta(LGDPPR_STRAINSVAR.Meta):
        verbose_name = u'Полное расписание. Вариант расписания|LGDPPR.STRAINSVAR'
        verbose_name_plural = u'Полное расписание. Варианты расписания|LGDPPR.STRAINSVAR'


class CurrentMixin(object):
    @classmethod
    def get_base_field_names(cls):
        return [
            f.name for f in cls._meta.fields
            if f.name != 'id'
        ]

    def update_from_update_object(self, other_obj):
        diffs = []
        for fn in self.get_base_field_names():
            if getattr(self, fn) != getattr(other_obj, fn):
                diffs.append({
                    'field': fn,
                    'from': getattr(self, fn),
                    'to': getattr(other_obj, fn)
                })
            setattr(self, fn, getattr(other_obj, fn))

        return diffs

    @classmethod
    def build_from_update_object(cls, other_obj):
        return cls(**{
            fn: getattr(other_obj, fn)
            for fn in cls.get_base_field_names()
        })

    @classmethod
    def create_from_update_object(cls, other_obj):
        obj = cls.build_from_update_object(other_obj)
        obj.save()

        return obj

    @classmethod
    def diff_by_base_fields(cls, obj1, obj2):
        diffs = []
        for fn in cls.get_base_field_names():
            if getattr(obj1, fn) != getattr(obj2, fn):
                diffs.append({
                    'field': fn,
                    'obj1': getattr(obj1, fn),
                    'obj2': getattr(obj2, fn)
                })

        return diffs


class Current_SCALENDAR(LGDPPR_SCALENDAR, CurrentMixin):
    class Meta(LGDPPR_SCALENDAR.Meta):
        verbose_name = u'Текущее расписание. Календарь поезда|LGDPPR.SCALENDAR'
        verbose_name_plural = u'Текущее расписание. Календари поезда|LGDPPR.SCALENDAR'


class Current_SDOCS(LGDPPR_SDOCS, CurrentMixin):
    pass


class Current_SRASPRP(LGDPPR_SRASPRP, CurrentMixin):
    class Meta(LGDPPR_SRASPRP.Meta):
        verbose_name = u'Текущее расписание. Расписание|LGDPPR.SRASPRP'
        verbose_name_plural = u'Текущее расписание. Расписания|LGDPPR.SRASPRP'


class Current_STRAINS(LGDPPR_STRAINS, CurrentMixin):
    class Meta(LGDPPR_STRAINS.Meta):
        verbose_name = u'Текущее расписание. Поезд|LGDPPR.STRAINS'
        verbose_name_plural = u'Текущее расписание. Поезда|LGDPPR.STRAINS'


class Current_STRAINSVAR(LGDPPR_STRAINSVAR, CurrentMixin):
    class Meta(LGDPPR_STRAINSVAR.Meta):
        verbose_name = u'Текущее расписание. Вариант расписания|LGDPPR.STRAINSVAR'
        verbose_name_plural = u'Текущее расписание. Варианты расписания|LGDPPR.STRAINSVAR'


class ChangeMixIn(object):
    def get_spec_buf(self):
        return Change_SPEC_BUF.objects.get(update=self.update, DATE_GVC=self.DATE_GVC,
                                           ID_XML=self.ID_XML)

    def get_current(self):
        raise NotImplementedError()

    @classmethod
    def get_by_spec_buf(cls, spec_buf):
        return cls.objects.filter(
            ID_XML=spec_buf.ID_XML, DATE_GVC=spec_buf.DATE_GVC, update=spec_buf.update
        )


class Change_SPEC_BUF(models.Model):
    update = models.ForeignKey('rzd_buffer.Update', null=False)
    IDTR = models.IntegerField(null=False, help_text=u'''
        Идентификатор поезда  или 0, если изменения вносились в буферные таблицы, не содержащие данные
        идентификаторов поездов.
    ''')
    KOP = models.IntegerField(null=False, help_text=mark_safe(u'''
        Вид операции над данными (вставка, замена, удаление) или операция завершения передачи корректировок
        для ведения таблицы полноты.<br>
        “01” вставить информацию по этому сообщению<br>
        “02” удалить информацию по заданному поезду и вставить из этого сообщения (замена)<br>
        “03” удалить информацию по этому сообщению.
    '''))
    KOP_INSERT_MODE = 1
    KOP_UPDATE_MODE = 2
    KOP_DELETE_MODE = 3
    VER = models.CharField(null=True, max_length=500)
    DT = models.CharField(null=True, max_length=500)
    USER = models.CharField(null=True, max_length=500)

    DATE_GVC = models.DateTimeField(null=False, help_text=u'''
        Дата и время внесения изменений в ГВЦ.
    ''')
    ID_XML = models.IntegerField(null=False, help_text=u'''
        Идентификатор XML-документа
    ''', db_index=True)
    SIST_IST = models.IntegerField(null=False)

    af_status = models.IntegerField(u'А.Ф. Статус', default=0, null=False, choices=(
        (0, u'Открыт'),
        (1, u'Важный'),
        (2, u'Решен'),
        (3, u'Проверен')
    ))

    def __str__(self):
        return u'<SPEC_BUF: KOP={self.KOP}, DATE_GVC={self.DATE_GVC}, ID_XML={self.ID_XML}>'.format(self=self)

    @property
    def key(self):
        return build_spec_buf_key(self)

    class Meta:
        verbose_name = u'Изменение|LGDPPR.SPEC_BUF'
        verbose_name_plural = u'Изменения|LGDPPR.SPEC_BUF'


class Change_STRAINS_BUF(LGDPPR_STRAINS, ChangeMixIn):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    DATE_GVC = models.DateTimeField(null=False, help_text=DATE_GVC_HELP_TEXT)
    ID_XML = models.IntegerField(null=False, help_text=ID_XML_HELP_TEXT)
    SIST_IST = models.IntegerField(null=False, help_text=SIST_IST_HELP_TEXT)

    class Meta(LGDPPR_STRAINS.Meta):
        verbose_name = u'Изменения. Поезд|LGDPPR.STRAINS'
        verbose_name_plural = u'Изменения. Поезда|LGDPPR.STRAINS'


class Change_STRAINSVAR_BUF(LGDPPR_STRAINSVAR, ChangeMixIn):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    DATE_GVC = models.DateTimeField(null=False, help_text=DATE_GVC_HELP_TEXT)
    ID_XML = models.IntegerField(null=False, help_text=ID_XML_HELP_TEXT)
    SIST_IST = models.IntegerField(null=False, help_text=SIST_IST_HELP_TEXT)

    class Meta(LGDPPR_STRAINSVAR.Meta):
        verbose_name = u'Изменения. Вариант расписания|LGDPPR.STRAINSVAR'
        verbose_name_plural = u'Изменения. Варианты расписания|LGDPPR.STRAINSVAR'


class Change_SRASPRP_BUF(LGDPPR_SRASPRP, ChangeMixIn):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    DATE_GVC = models.DateTimeField(null=False, help_text=DATE_GVC_HELP_TEXT)
    ID_XML = models.IntegerField(null=False, help_text=ID_XML_HELP_TEXT, db_index=True)
    SIST_IST = models.IntegerField(null=False, help_text=SIST_IST_HELP_TEXT)

    class Meta(LGDPPR_SRASPRP.Meta):
        verbose_name = u'Изменения. Расписание|LGDPPR.SRASPRP'
        verbose_name_plural = u'Изменения. Расписания|LGDPPR.SRASPRP'


class Change_SCALENDAR_BUF(LGDPPR_SCALENDAR, ChangeMixIn):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    DATE_GVC = models.DateTimeField(null=False, help_text=DATE_GVC_HELP_TEXT)
    ID_XML = models.IntegerField(null=False, help_text=ID_XML_HELP_TEXT, db_index=True)
    SIST_IST = models.IntegerField(null=False, help_text=SIST_IST_HELP_TEXT)

    class Meta(LGDPPR_SCALENDAR.Meta):
        verbose_name = u'Изменения. Календарь поезда|LGDPPR.SCALENDAR'
        verbose_name_plural = u'Изменения. Календари поезда|LGDPPR.SCALENDAR'


class Change_SDOCS_BUF(LGDPPR_SDOCS, ChangeMixIn):
    update = models.ForeignKey('rzd_buffer.Update', null=False)

    DATE_GVC = models.DateTimeField(null=False)
    ID_XML = models.IntegerField(null=False)
    SIST_IST = models.IntegerField(null=False)


class Marker(models.Model):
    """
    RASPADMIN-1189: Сделать таблицу для хранения метаинформации парсера РЖД
    """
    ID_XML = models.IntegerField(null=False, db_index=True)
    IDR = models.IntegerField(null=False, db_index=True)
    OP = models.IntegerField(null=False, db_index=True)
    status = models.IntegerField(null=True, blank=True, default=None)
    # @af1461: Это поля для записи информации о том, кто проводил верификацию того или иного изменения.
    # https://st.yandex-team.ru/RASPADMIN-1224
    user = models.IntegerField(null=True, blank=True, default=None)
    time = models.DateTimeField(null=True, blank=True, default=None)

    class Meta:
        db_table = 'markers'


class LVGD01_TR2PROC_feed_mysql(models.Model):
    """
    Копия коллекции common.apps.suburban_events.models.LVGD01_TR2PROC_feed, но не в mongo, а в mysql.
    Для работы сервиса она не требуется, нужна временно, чтобы @af1461 мог на первом этапе использовать
    свои mysql-скрипты для оценки данных РЖД.
    """

    ID_TRAIN = models.IntegerField(null=False, help_text=u'''
        Идентификатор отправившегося поезда - все его события будут иметь один ID_TRAIN.
        Когда этот же поезд отправится завтра - ID_TRAIN поменяется
        Оригинальное название колонки в базе РЖД - ID
    ''')
    IDTR = models.IntegerField(null=False, help_text=u'''
        Идентификатор поезда из НГДП
    ''')
    IDRASP = models.IntegerField(null=False, help_text=u'''
        Идентификатор расписания назначенного поезда из НГДП
    ''')
    STORASP = models.IntegerField(null=False, help_text=u'''
        Код станции отправления
    ''')
    STOEX = models.IntegerField(null=False, help_text=u'''
        Код станции отправления ЭКСПРЕСС
    ''')
    NAMESTO = models.CharField(null=False, max_length=100, help_text=u'''
        Наименование станции отправления
    ''')
    STNRASP = models.IntegerField(null=False, help_text=u'''
        Код станции назначения
    ''')
    STNEX = models.IntegerField(null=False, help_text=u'''
        Код станции назначения ЭКСПРЕСС
    ''')
    NAMESTN = models.CharField(null=False, max_length=100, help_text=u'''
        Наименование станции назначения
    ''')
    NOMPEX = models.CharField(null=False, max_length=100, help_text=u'''
        № поезда в системе ЭКСПРЕСС
    ''')
    NAMEP = models.CharField(null=False, max_length=100, help_text=u'''
        Наименование поезда
    ''')
    SOURCE = models.IntegerField(null=False, help_text=u'''
        Система – источник информации. 1 – АСОУП, 2 – ГИД Урал, 3 – САИ ПС, 4 – ГИС РЖД
    ''')
    KODOP = models.IntegerField(null=False, help_text=u'''
        Код операции с поездом. 1 – прибытие, 3 – отправление
    ''')
    DOR = models.IntegerField(null=False, help_text=u'''
        Код дороги ввода информации
    ''')
    OTD = models.IntegerField(null=False, help_text=u'''
        Код отделения
    ''')
    NOMRP = models.IntegerField(null=False, help_text=u'''
        Порядковый номер станции по ходу поезда
    ''')
    STOPER = models.IntegerField(null=False, help_text=u'''
        Код станции совершения операции
    ''')
    STOPEREX = models.IntegerField(null=False, help_text=u'''
        Код станции совершения операции ЭКСПРЕСС
    ''')
    STNAME = models.CharField(null=False, max_length=100, help_text=u'''
        Наименование станции совершения операции
    ''')
    TIMEOPER_N = models.DateTimeField(null=False, help_text=u'''
        Дата и время совершения операции с поездом из эталонного расписания
    ''')
    TIMEOPER_F = models.DateTimeField(null=False, help_text=u'''
        Дата и время совершения операции с поездом фактические
    ''')
    KM = models.IntegerField(null=False, help_text=u'''
        Километраж от начала маршрута
    ''')
    PRSTOP = models.FloatField(null=False, help_text=u'''
        Признак стоянки без посадки/высадки пассажиров.
        1-стоянка без посадки/высадки; 0–обычный режим
        Признак стоянки без посадки/высадки пассажиров означает следующее: если время стоянки не равно 0 и признак 0 –
        посадка/высадка есть, признак 1 – посадки/высадки нет; если время стоянки равно 0, признак не надо анализировать
    ''')
    PRIORITY = models.FloatField(null=False, help_text=u'''
        Приоритетная запись о фактическом маршруте
    ''')
    PRIORITY_RATING = models.FloatField(null=False, help_text=u'''
        Рейтинг системы-источника в таблице приоритетов
    ''')

    class Meta:
        verbose_name = u'Электрички realtime - лента (mysql)'
