# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import random
from contextlib import contextmanager
from threading import local
from functools import wraps

import ibm_db
from django.conf import settings

from common.dynamic_settings.core import DynamicSetting
from common.dynamic_settings.default import conf
from common.utils.try_hard import try_hard

from travel.rasp.suburban_tasks.suburban_tasks.rzd_hosts import get_rzd_hosts_and_ports

from .cpy_pst import PercentageStatus, chunker


log = logging.getLogger(__name__)


"""
Нужно сразу собирать все данные, чтобы коннект не отпал посреди вызова.
Такое почему то случается. Поэтому использования yield пока исключаем.
"""


conf.register_settings(
    RZD_SCHEDULE_GET_DATA_RETRY_DELAY=DynamicSetting(
        10, cache_time=60,
        description='Сколько ждем перед повторной попыткой получить данные расписаний от РЖД'),
    RZD_SCHEDULE_GET_DATA_MAX_RETRIES=DynamicSetting(
        10, cache_time=60,
        description='Максимальное количество попыток получить данные расписаний от РЖД'),
)


default_retrier = try_hard(
    max_retries=conf.RZD_SCHEDULE_GET_DATA_MAX_RETRIES,
    sleep_duration=conf.RZD_SCHEDULE_GET_DATA_RETRY_DELAY,
)


connection_store = local()

GVC_PROC_DATE_FORMAT = '%Y-%m-%d-%H.%M'


class ManagerRequiredError(Exception):
    pass


class ManagerAlreadyInitializedError(Exception):
    pass


def get_raw_connection(host, port, db_name):
    """
    Про ошибки коннекта читать тут
    https://www.ibm.com/support/knowledgecenter/en/SSEPGG_11.1.0/com.ibm.db2.luw.messages.sql.doc/doc/rsqlmsg.html

    :param host:
    :param db_name:
    :param port:
    :return:
    """

    connection_string = ';'.join([
        'DRIVER={IBM DB2 ODBC DRIVER}',
        'DATABASE={DATABASE_NAME}'.format(DATABASE_NAME=db_name),
        'HOSTNAME={HOSTNAME}'.format(HOSTNAME=host), 'PORT={}'.format(port),
        'PROTOCOL=TCPIP', 'UID=DGDPEX04', 'PWD=DGDP5396'
    ])
    log.info(u'Подключаемся к базе РЖД %s: %s', db_name, connection_string)
    connection = ibm_db.connect(connection_string, '', '')
    log.info(u'Успешно подключились к базе РЖД %s', db_name)
    log.info(u'Отключаем автокоммит')
    ibm_db.autocommit(connection, False)

    return connection


@contextmanager
def rzd_db_manager():
    try:
        if hasattr(connection_store, 'connection'):
            raise ManagerAlreadyInitializedError(u'Менеджер подключений для этой нитки уже инициализирован')
        connection_store.connection = None
        yield
    finally:
        if hasattr(connection_store, 'connection'):
            if ibm_db.active(connection_store.connection):
                ibm_db.close(connection_store.connection)
            del connection_store.connection


def use_rzd_db_manager(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        with rzd_db_manager():
            return func(*args, **kwargs)

    return wrapper


def get_connection_to_any_host(hosts_and_ports, db_name):
    log.info("Rzd proxy hosts: %s", hosts_and_ports)

    for host, port in hosts_and_ports:
        try:
            log.info("Trying to connect to rzd proxy %s:%s", host, port)
            connection = get_raw_connection(host, port, db_name)
        except Exception:
            log.exception("Can't connect to rzd host: {}:{}".format(host, port))
        else:
            return connection
    else:
        raise Exception("Can't connect to any of {} rzd hosts".format(hosts_and_ports))


def get_connect(db_name=None):
    if not hasattr(connection_store, 'connection'):
        raise ManagerRequiredError(u'Нужно использовать rzd_db_manager для управляемых подключений')

    if db_name is None:
        db_name = settings.RZD_DATABASE_NAME

    if not ibm_db.active(connection_store.connection):
        hosts_and_ports = get_rzd_hosts_and_ports()
        random.shuffle(hosts_and_ports)  # не лучший способ балансровки, но для наших ржд-скриптов - ок
        connection_store.connection = get_connection_to_any_host(hosts_and_ports, db_name)

    return connection_store.connection


def get_rowdicts_from_resource(resource):
    # Нужно сразу собрать все данные, чтобы коннект не отпал посреди вызова.
    # Такое почему-то случается.
    rowdicts = []

    while True:
        rowdict = ibm_db.fetch_assoc(resource)
        if not rowdict:
            break

        rowdicts.append(rowdict)

    return rowdicts


def rzd_sql_execute(connect, sql):
    log.info(u'Запускаем РЖД SQL: %s', sql)
    return ibm_db.exec_immediate(connect, sql)


def repr_call_proc_param(param):
    if isinstance(param, basestring):
        return u"'{}'".format(param)
    else:
        return unicode(param)


def rzd_sql_callproc(connect, procedure, params):
    log.info(u'Запускаем РЖД SQL PROCEDURE: %s (%s)', procedure, u', '.join(map(repr_call_proc_param, params)))
    return ibm_db.callproc(
        connect, procedure,
        params
    )


def get_gdpprbase_regions():
    connect = get_connect()
    rzd_sql_callproc(connect, 'LGDP13.GDPPRBASE', (3, '-1', 'SREG', '-1', -1, 0, 0, ''))

    resource = rzd_sql_execute(connect, 'SELECT * FROM LGDPPR.TMP_SREG')

    return get_rowdicts_from_resource(resource)


def get_gdpprbase_strains_for_region(region_CREG, today):
    connect = get_connect()
    results = []
    for year in (today.year - 1, today.year, today.year + 1):
        rzd_sql_callproc(
            connect, 'LGDP13.GDPPRBASE',
            (4, region_CREG, '-1', '-1', year, 0, 0, '')
        )
        resource = rzd_sql_execute(connect, 'SELECT * FROM LGDPPR.TMP_STRAINS')
        results.extend(get_rowdicts_from_resource(resource))

    return results


def get_trains_related_data(train_ids):
    status = PercentageStatus(len(train_ids), log)

    for id_chunk in chunker(train_ids, 20):
        # Т.к. это генератор, подключение может отвалиться, если выходные данные долго обрабатываем
        connect = get_connect()
        data = {
            'STRAINSVAR': [],
            'SRASPRP': [],
            'SCALENDAR': [],
            'SDOCS': [],
        }
        rzd_sql_callproc(
            connect, 'LGDP13.GDPPRBASE',
            (2, '-1', '-1', ','.join(map(str, id_chunk)), '-1', 0, 0, '')
        )

        for tbl_name, rowdicts in data.items():
            resource = rzd_sql_execute(connect, 'SELECT * FROM LGDPPR.TMP_{}'.format(tbl_name))
            rowdicts.extend(get_rowdicts_from_resource(resource))

        status.step(len(id_chunk))

        yield data


def get_stations():
    connect = get_connect()
    result = rzd_sql_callproc(
        connect, 'LNSI01.GETNSI',
        ('IC00.STAN', '1970-01-01-00.00.00', '9999-12-31-23.00.00', 0)
    )

    resource = result[0]
    return get_rowdicts_from_resource(resource)


@default_retrier
def get_changes_spec_buf_rows(query_from_dt, query_to_dt):
    connect = get_connect()
    rzd_sql_callproc(
        connect, 'LGDP13.GDPPRCOR',
        (1,
         query_from_dt.strftime(GVC_PROC_DATE_FORMAT),
         query_to_dt.strftime(GVC_PROC_DATE_FORMAT),
         '-1', 0, 0, '')
    )

    resource = rzd_sql_execute(connect, 'SELECT * FROM LGDPPR.TMP_SPEC_BUF')
    rows = get_rowdicts_from_resource(resource)
    ibm_db.commit(connect)

    rows.sort(key=lambda r: r['DATE_GVC'])
    return rows


@default_retrier
def get_train_changes_related_data(train_ids, query_from_dt, query_to_dt):
    connect = get_connect()
    status = PercentageStatus(len(train_ids), log)

    data = {
        'STRAINSVAR': [],
        'SRASPRP': [],
        'SCALENDAR': [],
        'SDOCS': [],
        'STRAINS': [],
    }

    for id_chunk in chunker(train_ids, 20):
        rzd_sql_callproc(
            connect, 'LGDP13.GDPPRCOR',
            (2,
             query_from_dt.strftime(GVC_PROC_DATE_FORMAT),
             query_to_dt.strftime(GVC_PROC_DATE_FORMAT),
             ','.join(map(str, id_chunk)),
             0, 0, '')
        )

        for tbl_name, rowdicts in data.items():
            resource = rzd_sql_execute(connect, 'SELECT * FROM LGDPPR.TMP_{}_BUF'.format(tbl_name))
            rowdicts.extend(get_rowdicts_from_resource(resource))

        ibm_db.commit(connect)

        status.step(len(id_chunk))

    return data
